/**
 * GzipUtils.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 * Copyright (C) 2003-2009 www.devshed.com
 * Copyright (C) 1994-2009 Sun Microsystems, Inc.
 */
package adams.core.io;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import adams.core.annotation.MixedCopyright;

/**
 * Helper class for gzip related operations.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3755 $
 */
@MixedCopyright
public class GzipUtils {

  /** the default extension. */
  public final static String EXTENSION = ".gz";

  /**
   * Decompresses the specified gzip archive to a file without the ".gz"
   * extension.
   *
   * @param archiveFile	the gzip file to decompress
   * @param buffer	the buffer size to use
   * @return		the error message, null if everything OK
   */
  public static String decompress(File archiveFile, int buffer) {
    return decompress(archiveFile, buffer, new PlaceholderFile(archiveFile.getAbsolutePath().replaceAll("\\.gz$", "")));
  }

  /**
   * Decompresses the specified gzip archive.
   * <p/>
   * See <a href="http://java.sun.com/developer/technicalArticles/Programming/compression/" target="_blank">Compressing and Decompressing Data Using Java APIs</a>.
   *
   * @param archiveFile	the gzip file to decompress
   * @param buffer	the buffer size to use
   * @param outputFile	the destination file
   * @return		the error message, null if everything OK
   */
  public static String decompress(File archiveFile, int buffer, File outputFile) {
    String			result;
    byte[] 			buf;
    int 			len;
    GZIPInputStream 		in;
    BufferedOutputStream 	out;
    String			msg;

    in     = null;
    out    = null;
    result = null;
    try {

      // does file already exist?
      if (outputFile.exists())
	System.err.println("WARNING: overwriting '" + outputFile + "'!");

      // create GZIP file
      buf = new byte[buffer];
      in  = new GZIPInputStream(new BufferedInputStream(new FileInputStream(archiveFile.getAbsolutePath())));
      out = new BufferedOutputStream(new FileOutputStream(outputFile), buffer);

      // Transfer bytes from the file to the GZIP file
      while ((len = in.read(buf)) > 0)
	out.write(buf, 0, len);

      // finish up
      in.close();
      in = null;
      out.close();
      out = null;
    }
    catch (Exception e) {
      msg = "Failed to decompress '" + archiveFile + "': ";
      System.err.println(msg);
      e.printStackTrace();
      result = msg + e;
    }
    finally {
      if (in != null) {
	try {
	  in.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
      if (out != null) {
	try {
	  out.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }

  /**
   * Compresses the specified gzip archive to a file with the ".gz"
   * extension.
   *
   * @param inputFile	the gzip file to compress
   * @param buffer	the buffer size to use
   * @return		the error message, null if everything OK
   */
  public static String compress(File inputFile, int buffer) {
    return compress(inputFile, buffer, new PlaceholderFile(inputFile.getAbsolutePath() + EXTENSION));
  }

  /**
   * Compresses the specified gzip archive. Does not remove the input file.
   *
   * @param inputFile	the gzip file to compress
   * @param buffer	the buffer size to use
   * @param outputFile	the destination file (the archive)
   * @return		the error message, null if everything OK
   */
  public static String compress(File inputFile, int buffer, File outputFile) {
    return compress(inputFile, buffer, outputFile, false);
  }

  /**
   * Compresses the specified gzip archive.
   * <p/>
   * See <a href="http://www.devshed.com/c/a/Java/GZIPping-with-Java/2/" target="_blank">GZIPping with Java</a>.
   *
   * @param inputFile	the gzip file to compress
   * @param buffer	the buffer size to use
   * @param outputFile	the destination file (the archive)
   * @param removeInput	whether to remove the input file
   * @return		the error message, null if everything OK
   */
  public static String compress(File inputFile, int buffer, File outputFile, boolean removeInput) {
    String		result;
    byte[] 		buf;
    int 		len;
    GZIPOutputStream 	out;
    BufferedInputStream in;
    String		msg;

    in     = null;
    out    = null;
    result = null;
    try {
      // does file already exist?
      if (outputFile.exists())
	System.err.println("WARNING: overwriting '" + outputFile + "'!");

      // create GZIP file
      buf = new byte[buffer];
      out = new GZIPOutputStream(new FileOutputStream(outputFile));
      in  = new BufferedInputStream(new FileInputStream(inputFile.getAbsolutePath()));

      // Transfer bytes from the file to the GZIP file
      while ((len = in.read(buf)) > 0)
	out.write(buf, 0, len);

      // finish up
      in.close();
      in = null;
      out.close();
      out = null;

      // remove input file?
      if (removeInput) {
	if (!inputFile.delete())
	  result = "Failed to delete input file '" + inputFile + "' after successful compression!";
      }
    }
    catch (Exception e) {
      msg = "Failed to decompress '" + inputFile + "': ";
      System.err.println(msg);
      e.printStackTrace();
      result = msg + e;
    }
    finally {
      if (in != null) {
	try {
	  in.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
      if (out != null) {
	try {
	  out.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }
}
