/**
 * ZipUtils.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 * Copyright (C) 2005-2008 www.java-tips.org
 * Copyright (C) 1994-2009 Sun Microsystems, Inc.
 */
package adams.core.io;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.Enumeration;
import java.util.Vector;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import adams.core.annotation.MixedCopyright;
import adams.core.base.BaseRegExp;

/**
 * A helper class for ZIP-file related tasks.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3763 $
 */
@MixedCopyright
public class ZipUtils {

  /**
   * Creates a zip file from the specified files.
   *
   * @param output	the output file to generate
   * @param files	the files to store in the zip file
   * @return		null if successful, otherwise error message
   * @see		#compress(File, File[], int)
   */
  public static String compress(File output, File[] files) {
    return compress(output, files, 1024);
  }

  /**
   * Creates a zip file from the specified files.
   *
   * @param output	the output file to generate
   * @param files	the files to store in the zip file
   * @param bufferSize	the buffer size to use
   * @return		null if successful, otherwise error message
   * @see		#compress(File, File[], String, int)
   */
  public static String compress(File output, File[] files, int bufferSize) {
    return compress(output, files, "", bufferSize);
  }

  /**
   * Creates a zip file from the specified files.
   * <p/>
   * See <a href="http://www.java-tips.org/java-se-tips/java.util.zip/how-to-create-a-zip-file.html" target="_blank">How to create a ZIP File</a>.
   *
   * @param output	the output file to generate
   * @param files	the files to store in the zip file
   * @param stripRegExp	the regular expression used to strip the file names
   * @param bufferSize	the buffer size to use
   * @return		null if successful, otherwise error message
   */
  public static String compress(File output, File[] files, String stripRegExp, int bufferSize) {
    String			result;
    int				i;
    byte[] 			buf;
    int 			len;
    ZipOutputStream 		out;
    BufferedInputStream 	in;
    String			filename;
    String			msg;

    in     = null;
    out    = null;
    result = null;
    try {
      // does file already exist?
      if (output.exists())
	System.err.println("WARNING: overwriting '" + output + "'!");

      // create ZIP file
      buf = new byte[bufferSize];
      out = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(output.getAbsolutePath())));
      for (i = 0; i < files.length; i++) {
	in = new BufferedInputStream(new FileInputStream(files[i].getAbsolutePath()));

	// Add ZIP entry to output stream.
	filename = files[i].getParentFile().getAbsolutePath();
	if (stripRegExp.length() > 0)
	  filename = filename.replaceFirst(stripRegExp, "");
	if (filename.length() > 0)
	  filename += File.separator;
	filename += files[i].getName();
	out.putNextEntry(new ZipEntry(filename));

	// Transfer bytes from the file to the ZIP file
	while ((len = in.read(buf)) > 0)
	  out.write(buf, 0, len);

	// Complete the entry
	out.closeEntry();
	in.close();
	in = null;
      }

      // Complete the ZIP file
      out.close();
      out = null;
    }
    catch (Exception e) {
      msg = "Failed to generate archive '" + output + "': ";
      System.err.println(msg);
      e.printStackTrace();
      result = msg + e;
    }
    finally {
      if (in != null) {
	try {
	  in.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
      if (out != null) {
	try {
	  out.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }

  /**
   * Unzips the files in a ZIP file. Does not recreate the directory structure
   * stored in the ZIP file.
   *
   * @param input	the ZIP file to unzip
   * @param outputDir	the directory where to store the extracted files
   * @return		the successfully extracted files
   * @see		#decompress(File, File, boolean)
   */
  public static Vector<File> decompress(File input, File outputDir) {
    return decompress(input, outputDir, false);
  }

  /**
   * Unzips the files in a ZIP file.
   *
   * @param input	the ZIP file to unzip
   * @param outputDir	the directory where to store the extracted files
   * @param createDirs	whether to re-create the directory structure from the
   * 			ZIP file
   * @return		the successfully extracted files
   * @see		#decompress(File, File, boolean, String, boolean)
   */
  public static Vector<File> decompress(File input, File outputDir, boolean createDirs) {
    return decompress(input, outputDir, createDirs, new BaseRegExp(""), false);
  }

  /**
   * Unzips the files in a ZIP file. Files can be filtered based on their
   * filename, using a regular expression (the matching sense can be inverted).
   *
   * @param input	the ZIP file to unzip
   * @param outputDir	the directory where to store the extracted files
   * @param createDirs	whether to re-create the directory structure from the
   * 			ZIP file
   * @param match	the regular expression that the files are matched against
   * @param invertMatch	whether to invert the matching sense
   * @return		the successfully extracted files
   * @see		#decompress(File, File, boolean, String, boolean, int)
   */
  public static Vector<File> decompress(File input, File outputDir, boolean createDirs, BaseRegExp match, boolean invertMatch) {
    return decompress(input, outputDir, createDirs, match, invertMatch, 1024);
  }

  /**
   * Unzips the files in a ZIP file. Files can be filtered based on their
   * filename, using a regular expression (the matching sense can be inverted).
   * <p/>
   * See <a href="http://java.sun.com/developer/technicalArticles/Programming/compression/" target="_blank">Compressing and Decompressing Data Using Java APIs</a>.
   *
   * @param input	the ZIP file to unzip
   * @param outputDir	the directory where to store the extracted files
   * @param createDirs	whether to re-create the directory structure from the
   * 			ZIP file
   * @param match	the regular expression that the files are matched against
   * @param invertMatch	whether to invert the matching sense
   * @param bufferSize	the buffer size to use
   * @return		the successfully extracted files
   */
  public static Vector<File> decompress(File input, File outputDir, boolean createDirs, BaseRegExp match, boolean invertMatch, int bufferSize) {
    return decompress(input, outputDir, createDirs, match, invertMatch, bufferSize, new StringBuilder());
  }

  /**
   * Unzips the files in a ZIP file. Files can be filtered based on their
   * filename, using a regular expression (the matching sense can be inverted).
   * <p/>
   * See <a href="http://java.sun.com/developer/technicalArticles/Programming/compression/" target="_blank">Compressing and Decompressing Data Using Java APIs</a>.
   *
   * @param input	the ZIP file to unzip
   * @param outputDir	the directory where to store the extracted files
   * @param createDirs	whether to re-create the directory structure from the
   * 			ZIP file
   * @param match	the regular expression that the files are matched against
   * @param invertMatch	whether to invert the matching sense
   * @param bufferSize	the buffer size to use
   * @param errors	for storing potential errors
   * @return		the successfully extracted files
   */
  public static Vector<File> decompress(File input, File outputDir, boolean createDirs, BaseRegExp match, boolean invertMatch, int bufferSize, StringBuilder errors) {
    Vector<File>		result;
    ZipFile			archive;
    Enumeration<ZipEntry>	enm;
    ZipEntry			entry;
    File			outFile;
    String			outName;
    byte[]			buffer;
    BufferedInputStream		in;
    BufferedOutputStream	out;
    int				len;
    String			error;

    result  = new Vector<File>();
    archive = null;
    try {
      // unzip archive
      buffer  = new byte[bufferSize];
      archive = new ZipFile(input.getAbsoluteFile());
      enm     = (Enumeration<ZipEntry>) archive.entries();
      while (enm.hasMoreElements()) {
	entry = enm.nextElement();

	if (entry.isDirectory() && !createDirs)
	  continue;

	// does name match?
	if (!match.isMatchAll() && !match.isEmpty()) {
	  if (invertMatch && match.isMatch(entry.getName()))
	    continue;
	  else if (!invertMatch && !match.isMatch(entry.getName()))
	    continue;
	}

	// extract
	if (entry.isDirectory() && createDirs) {
	  outFile = new File(outputDir.getAbsolutePath() + File.separator + entry.getName());
	  if (!outFile.mkdirs()) {
	    error = "Failed to create directory '" + outFile.getAbsolutePath() + "'!";
	    System.err.println(error);
	    errors.append(error + "\n");
	  }
	}
	else {
	  in      = null;
	  out     = null;
	  outName = null;
	  try {
	    // assemble output name
	    outName = outputDir.getAbsolutePath() + File.separator;
	    if (createDirs)
	      outName += entry.getName();
	    else
	      outName += new File(entry.getName()).getName();

	    // create directory, if necessary
	    outFile = new File(outName).getParentFile();
	    if (!outFile.exists()) {
	      if (!outFile.mkdirs()) {
		error =
		    "Failed to create directory '" + outFile.getAbsolutePath() + "', "
		    + "skipping extraction of '" + outName + "'!";
		System.err.println(error);
		errors.append(error + "\n");
		continue;
	      }
	    }

	    // extract data
	    in  = new BufferedInputStream(archive.getInputStream(entry));
	    out = new BufferedOutputStream(new FileOutputStream(outName), bufferSize);
	    while ((len = in.read(buffer)) != -1)
	      out.write(buffer, 0, len);
	    result.add(new File(outName));
	  }
	  catch (Exception e) {
	    error = "Error extracting '" + entry.getName() + "' to '" + outName + "': " + e;
	    System.err.println(error);
	    errors.append(error + "\n");
	  }
	  finally {
	    if (in != null) {
	      try {
		in.close();
	      }
	      catch (Exception e) {
		// ignored
	      }
	    }
	    if (out != null) {
	      try {
		out.flush();
		out.close();
	      }
	      catch (Exception e) {
		// ignored
	      }
	    }
	  }
	}
      }
    }
    catch (Exception e) {
      e.printStackTrace();
      errors.append("Error occurred: " + e + "\n");
    }
    finally {
      if (archive != null) {
	try {
	  archive.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }

  /**
   * Unzips the specified from a ZIP file. Does not create any directories
   * in case the parent directories of "output" don't exist yet.
   *
   * @param input	the ZIP file to unzip
   * @param archiveFile	the file from the archive to extract
   * @param output	the name of the output file
   * @return		the successfully extracted files
   */
  public static boolean decompress(File input, String archiveFile, File output) {
    return decompress(input, archiveFile, output, false);
  }

  /**
   * Unzips the specified from a ZIP file.
   *
   * @param input	the ZIP file to unzip
   * @param archiveFile	the file from the archive to extract
   * @param output	the name of the output file
   * @param createDirs	whether to create the directory structure represented
   * 			by output file
   * @return		the successfully extracted files
   */
  public static boolean decompress(File input, String archiveFile, File output, boolean createDirs) {
    return decompress(input, archiveFile, output, createDirs, 1024, new StringBuilder());
  }

  /**
   * Unzips the specified from a ZIP file.
   *
   * @param input	the ZIP file to unzip
   * @param archiveFile	the file from the archive to extract
   * @param output	the name of the output file
   * @param createDirs	whether to create the directory structure represented
   * 			by output file
   * @param bufferSize	the buffer size to use
   * @param errors	for storing potential errors
   * @return		the successfully extracted files
   */
  public static boolean decompress(File input, String archiveFile, File output, boolean createDirs, int bufferSize, StringBuilder errors) {
    boolean			result;
    ZipFile			zipfile;
    Enumeration<ZipEntry>	enm;
    ZipEntry			entry;
    File			outFile;
    String			outName;
    byte[]			buffer;
    BufferedInputStream		in;
    BufferedOutputStream	out;
    int				len;
    String			error;

    result  = false;
    zipfile = null;
    try {
      // unzip archive
      buffer  = new byte[bufferSize];
      zipfile = new ZipFile(input.getAbsoluteFile());
      enm     = (Enumeration<ZipEntry>) zipfile.entries();
      while (enm.hasMoreElements()) {
	entry = enm.nextElement();

	if (entry.isDirectory())
	  continue;
	if (!entry.getName().equals(archiveFile))
	  continue;

	in      = null;
	out     = null;
	outName = null;
	try {
	  // output name
	  outName = output.getAbsolutePath();

	  // create directory, if necessary
	  outFile = new File(outName).getParentFile();
	  if (!outFile.exists()) {
	    if (!createDirs) {
		error =
		  "Output directory '" + outFile.getAbsolutePath() + " does not exist', "
		  + "skipping extraction of '" + outName + "'!";
		System.err.println(error);
		errors.append(error + "\n");
		break;
	    }
	    else {
	      if (!outFile.mkdirs()) {
		error =
		  "Failed to create directory '" + outFile.getAbsolutePath() + "', "
		  + "skipping extraction of '" + outName + "'!";
		System.err.println(error);
		errors.append(error + "\n");
		break;
	      }
	    }
	  }

	  // extract data
	  in  = new BufferedInputStream(zipfile.getInputStream(entry));
	  out = new BufferedOutputStream(new FileOutputStream(outName), bufferSize);
	  while ((len = in.read(buffer)) != -1)
	    out.write(buffer, 0, len);

	  result = true;
	}
	catch (Exception e) {
	  result = false;
	  error  = "Error extracting '" + entry.getName() + "' to '" + outName + "': " + e;
	  System.err.println(error);
	  errors.append(error + "\n");
	}
	finally {
	  if (in != null) {
	    try {
	      in.close();
	    }
	    catch (Exception e) {
	      // ignored
	    }
	  }
	  if (out != null) {
	    try {
	      out.flush();
	      out.close();
	    }
	    catch (Exception e) {
	      // ignored
	    }
	  }
	}
      }
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
      errors.append("Error occurred: " + e + "\n");
    }
    finally {
      if (zipfile != null) {
	try {
	  zipfile.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }

  /**
   * Lists the files stored in the ZIP file. Lists directories automatically.
   *
   * @param input	the ZIP file to obtain the file list from
   * @return		the stored files
   */
  public static Vector<File> listFiles(File input) {
    return listFiles(input, true);
  }

  /**
   * Lists the files stored in the ZIP file.
   *
   * @param input	the ZIP file to obtain the file list from
   * @param listDirs	whether to include directories in the list
   * @return		the stored files
   */
  public static Vector<File> listFiles(File input, boolean listDirs) {
    Vector<File>		result;
    ZipFile			zipfile;
    Enumeration<ZipEntry>	enm;
    ZipEntry			entry;

    result  = new Vector<File>();
    zipfile = null;
    try {
      zipfile = new ZipFile(input.getAbsoluteFile());
      enm     = (Enumeration<ZipEntry>) zipfile.entries();
      while (enm.hasMoreElements()) {
	entry = enm.nextElement();

	// extract
	if (entry.isDirectory() && listDirs) {
	  result.add(new File(entry.getName()));
	}
	else {
	  result.add(new File(entry.getName()));
	}
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    finally {
      if (zipfile != null) {
	try {
	  zipfile.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }
}
