/**
 * Index.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.io.Serializable;

/**
 * A simple class that translates human-readable 1-based index strings
 * (including "first", "second", "third", "last_2", "last_1" and "last")
 * into integer indices.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3117 $
 */
public class Index
  implements Serializable, CustomDisplayStringProvider, Comparable<Index>, ExampleProvider {

  /** for serialization. */
  private static final long serialVersionUID = 4295722716846349301L;

  /** the special index for "first". */
  public final static String FIRST = "first";

  /** the special string "second". */
  public final static String SECOND = "second";

  /** the special string "third". */
  public final static String THIRD = "third";

  /** the special string "last_1" (2nd to last). */
  public final static String LAST_1 = "last_1";

  /** the special string "last_2" (3rd to last). */
  public final static String LAST_2 = "last_2";

  /** the special index for "last". */
  public final static String LAST = "last";

  /** the underlying index. */
  protected String m_Index;

  /** the parsed integer index. */
  protected Integer m_IntIndex;

  /** the maximum number for the 1-based index. */
  protected int m_Max;

  /**
   * Initializes with no index.
   */
  public Index() {
    this("");
  }

  /**
   * Initializes with the given index, but no maximum.
   *
   * @param index	the index to use
   */
  public Index(String index) {
    this(index, -1);
  }

  /**
   * Initializes with the given index and maximum.
   *
   * @param index	the index to use
   * @param max		the maximum of the 1-based index (e.g., use "10" to
   * 			allow "1-10" or -1 for uninitialized)
   */
  public Index(String index, int max) {
    super();

    setIndex(index);
    setMax(max);
  }

  /**
   * Sets the index.
   *
   * @param value	the index to use
   */
  public void setIndex(String value) {
    m_Index    = clean(value);
    m_IntIndex = null;
  }

  /**
   * Returns the currently set index.
   *
   * @return		the index in use
   */
  public String getIndex() {
    return m_Index;
  }

  /**
   * Sets the maximum (1-max will be allowed).
   *
   * @param value	the maximum for the 1-based index
   */
  public void setMax(int value) {
    if (value <= 0)
      m_Max = -1;
    else
      m_Max = value;
    m_IntIndex = null;
  }

  /**
   * Returns the maximum.
   *
   * @return		the maximum for the 1-based index
   */
  public int getMax() {
    return m_Max;
  }

  /**
   * Checks whether a valid index has been supplied.
   *
   * @return		true if a valid index is available
   */
  public boolean hasIndex() {
    return (m_Index.length() > 0);
  }

  /**
   * Cleanses the given string. Only allows "first", "last" and numbers.
   *
   * @param s		the string to clean
   * @return		the cleansed string, "" if invalid one provided
   */
  protected String clean(String s) {
    String	tmp;
    String	result;
    int		num;

    result = "";
    tmp    = s.toLowerCase();
    if (tmp.equals(FIRST)) {
      result = tmp;
    }
    else if (tmp.equals(SECOND)) {
      result = tmp;
    }
    else if (tmp.equals(THIRD)) {
      result = tmp;
    }
    else if (tmp.equals(LAST_2)) {
      result = tmp;
    }
    else if (tmp.equals(LAST_1)) {
      result = tmp;
    }
    else if (tmp.equals(LAST)) {
      result = tmp;
    }
    else {
      try {
	num = Integer.parseInt(tmp);
	if (num > 0)
	  result = tmp;
      }
      catch (Exception e) {
	// ignored
      }
    }

    return result;
  }

  /**
   * Parses the string and checks it against the maximum.
   *
   * @param s		the string to parse
   * @param max		the maximum to allow
   * @return		the parsed value, -1 if invalid
   */
  protected int parse(String s, int max) {
    int		result;

    result = -1;

    if (max > -1) {
      if (s.equals(FIRST)) {
	result = 0;
      }
      else if (s.equals(SECOND)) {
	result = 1;
      }
      else if (s.equals(THIRD)) {
	result = 2;
      }
      else if (s.equals(LAST_2)) {
	result = max - 3;
      }
      else if (s.equals(LAST_1)) {
	result = max - 2;
      }
      else if (s.equals(LAST)) {
	result = max - 1;
      }
      else {
	try {
	  result = Integer.parseInt(s) - 1;
	}
	catch (Exception e) {
	  // ignored
	  result = -1;
	}
      }

      // check boundaries
      if ((result > max - 1) || (result < 0))
	result = -1;
    }
    else {
      result = -1;
    }

    return result;
  }

  /**
   * Returns the integer representation of the index.
   *
   * @return		the integer index, -1 if not possible
   */
  public int getIntIndex() {
    if (m_IntIndex == null)
      m_IntIndex = parse(m_Index, m_Max);

    return m_IntIndex;
  }

  /**
   * Compares this index with the specified index for order. Returns a
   * negative integer, zero, or a positive integer as this index is less
   * than, equal to, or greater than the specified index.
   *
   * @param   o the subrange to be compared.
   * @return  a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   */
  public int compareTo(Index o) {
    int		result;

    if ((getMax() != -1) && (getMax() != -1)) {
      result = new Integer(getIntIndex()).compareTo(
	  new Integer(o.getIntIndex()));
    }
    else {
      result = new Integer(parse(getIndex(), Integer.MAX_VALUE)).compareTo(
	  new Integer(parse(o.getIndex(), Integer.MAX_VALUE)));
    }

    return result;
  }

  /**
   * Indicates whether some other object is "equal to" this one.
   *
   * @param obj		the reference object with which to compare.
   * @return		true if this object is the same as the obj argument;
   * 			false otherwise.
   */
  public boolean equals(Object obj) {
    if (!(obj instanceof Index))
      return false;
    else
      return (compareTo((Index) obj) == 0);
  }

  /**
   * Returns a string representation of the index object.
   *
   * @return		the representation
   */
  public String toString() {
    return "index=" + m_Index + ", max=" + m_Max;
  }

  /**
   * Returns the custom display string.
   *
   * @return		the string
   */
  public String toDisplay() {
    return getIndex();
  }

  /**
   * Returns the example.
   *
   * @return		the example
   */
  public String getExample() {
    return
        "An index is a number starting with 1; the following placeholders can be used as well: "
      + FIRST + ", " + SECOND + ", " + THIRD + ", " + LAST_2 + ", " + LAST_1 + ", " + LAST;
  }
}
