/**
 * NestedProducer.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Date;
import java.util.Enumeration;
import java.util.Vector;

import adams.core.DateFormat;
import adams.core.management.Java;
import adams.env.Environment;
import adams.env.Revisions;

/**
 * Generates the nested format.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4215 $
 */
public class NestedProducer
  extends AbstractRecursiveOptionProducer<Vector,Vector>
  implements BlacklistedOptionProducer {

  /** for serialization. */
  private static final long serialVersionUID = 7096746086060792830L;

  /** the line comment character in files storing nested option handlers. */
  public final static String COMMENT = "#";

  /** blacklisted classes. */
  protected Class[] m_Blacklisted;

  /** whether to suppress the prolog. */
  protected boolean m_OutputProlog;

  /** whether to print the classpath. */
  protected boolean m_OutputClasspath;

  /** for formatting dates. */
  protected static DateFormat m_DateFormat;
  static {
    m_DateFormat = new DateFormat("yyyy-MM-dd HH:mm:ss");
  }

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Generates the nested format (tab indentation in string representation, nested Vector objects in object representation).";
  }

  /**
   * Used for initializing members.
   */
  protected void initialize() {
    super.initialize();

    m_Blacklisted     = new Class[0];
    m_OutputProlog    = true;
    m_OutputClasspath = false;
  }

  /**
   * Sets the classes to avoid.
   *
   * @param value	the classes
   */
  public void setBlacklisted(Class[] value) {
    if (value == null)
      value = new Class[0];
    m_Blacklisted = value;
  }

  /**
   * Returns the blacklisted classes.
   *
   * @return		the classes
   */
  public Class[] getBlacklisted() {
    return m_Blacklisted;
  }

  /**
   * Sets whether to output the prolog (in comments) or not.
   *
   * @param value	if true then the prolog is generated
   */
  public void setOutputProlog(boolean value) {
    m_OutputProlog = value;
  }

  /**
   * Returns whether the prolog (comments) is generated.
   *
   * @return		true if the prolog is generated
   */
  public boolean getOutputProlog() {
    return m_OutputProlog;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputPrologTipText() {
    return "Whether to output the prolog with comments about software version, date/time of generation, etc.";
  }

  /**
   * Sets whether to output the classpath (in comments) or not.
   *
   * @param value	if true then the classpath is generated
   */
  public void setOutputClasspath(boolean value) {
    m_OutputClasspath = value;
  }

  /**
   * Returns whether the classpath (comments) is generated.
   *
   * @return		true if the classpath is generated
   */
  public boolean getOutputClasspath() {
    return m_OutputClasspath;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputClasspathTipText() {
    return "Whether to output the classpath in the comments as well.";
  }

  /**
   * Initializes the output data structure.
   *
   * @return		the created data structure
   */
  protected Vector initOutput() {
    return new Vector();
  }

  /**
   * Visits a boolean option.
   *
   * @param option	the boolean option
   * @return		the last internal data structure that was generated
   */
  public Vector processOption(BooleanOption option) {
    Vector	result;
    Object	currValue;

    result = new Vector();

    try {
      currValue = getCurrentValue(option);
      if (!((Boolean) currValue).equals((Boolean) option.getDefaultValue()))
	result.add(getOptionIdentifier(option));
    }
    catch (Exception e) {
      System.err.println("Error obtaining current value for '" + option.getProperty() + "':");
      e.printStackTrace();
    }

    if (result.size() > 0) {
      if (m_Nesting.empty())
	m_Output.addAll(result);
      else
	((Vector) m_Nesting.peek()).addAll(result);
    }

    return result;
  }

  /**
   * Visits a class option.
   *
   * @param option	the class option
   * @return		the last internal data structure that was generated
   */
  public Vector processOption(ClassOption option) {
    Vector			result;
    Object			currValue;
    Object			currValues;
    Object			value;
    int				i;
    Vector			nested;
    Vector			nestedDeeper;
    AbstractCommandLineHandler	handler;

    result = new Vector();

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      result.add(getOptionIdentifier(option));
      result.add(option.getVariable());
    }
    else {
      currValue = getCurrentValue(option);

      if (!isDefaultValue(option, currValue)) {
	currValues = null;

	if (currValue != null) {
	  if (!option.isMultiple()) {
	    currValues = Array.newInstance(option.getBaseClass(), 1);
	    Array.set(currValues, 0, currValue);
	  }
	  else {
	    currValues = currValue;
	  }

	  for (i = 0; i < Array.getLength(currValues); i++) {
	    value = Array.get(currValues, i);
	    result.add(getOptionIdentifier(option));
	    nested = new Vector();
	    result.add(nested);
	    nested.add(value.getClass().getName());
	    nestedDeeper = new Vector();
	    nested.add(nestedDeeper);
	    if (value instanceof OptionHandler) {
	      m_Nesting.push(nested);
	      m_Nesting.push(nestedDeeper);
	      doProduce(((OptionHandler) value).getOptionManager());
	      m_Nesting.pop();
	      m_Nesting.pop();
	    }
	    else {
	      handler = AbstractCommandLineHandler.getHandler(value);
	      nestedDeeper.addAll(Arrays.asList(handler.getOptions(value)));
	    }
	  }
	}
      }
    }

    if (m_Nesting.empty())
      m_Output.addAll(result);
    else
      ((Vector) m_Nesting.peek()).addAll(result);

    return result;
  }

  /**
   * Visits an argument option.
   *
   * @param option	the argument option
   * @return		the last internal data structure that was generated
   */
  public Vector processOption(AbstractArgumentOption option) {
    Vector	result;
    Object	currValue;
    Object	currValues;
    int		i;

    result = new Vector();

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      result.add(getOptionIdentifier(option));
      result.add(option.getVariable());
    }
    else {
      currValue = getCurrentValue(option);

      if (!isDefaultValue(option, currValue)) {
	currValues = null;

	if (currValue != null) {
	  if (!option.isMultiple()) {
	    currValues = Array.newInstance(option.getBaseClass(), 1);
	    Array.set(currValues, 0, currValue);
	  }
	  else {
	    currValues = currValue;
	  }

	  for (i = 0; i < Array.getLength(currValues); i++) {
	    result.add(getOptionIdentifier(option));
	    result.add(option.toString(Array.get(currValues, i)));
	  }
	}
      }
    }

    if (result.size() > 0) {
      if (m_Nesting.empty())
	m_Output.addAll(result);
      else
	((Vector) m_Nesting.peek()).addAll(result);
    }

    return result;
  }

  /**
   * Returns the indentation string for the given level.
   *
   * @param level	the level to generate the indentation string for
   * @return		the generated string
   */
  protected String getIndentation(int level) {
    StringBuffer	result;
    int			i;

    result = new StringBuffer();
    for (i = 0; i < level; i++)
      result.append("\t");

    return result.toString();
  }

  /**
   * Turns the nested options from an option handler into indentated lines.
   *
   * @param nested	the nested structure to turn into indentated lines
   * @param lines	the lines so far
   * @param level	the current level of indentation
   */
  protected void nestedToLines(Vector nested, Vector<String> lines, int level) {
    int		i;

    for (i = 0; i < nested.size(); i++) {
      if (nested.get(i) instanceof String)
        lines.add(getIndentation(level) + nested.get(i));
      else
        nestedToLines((Vector) nested.get(i), lines, level + 1);
    }
  }

  /**
   * Turns the nested options from an option handler into indentated lines.
   *
   * @param nested	the nested structure to turn into indentated lines
   * @return		the indentated lines
   */
  public Vector<String> nestedToLines(Vector nested) {
    Vector<String>	result;

    result = new Vector<String>();
    nestedToLines(nested, result, 0);

    return result;
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output, null in case of an error
   */
  public String toString() {
    StringBuilder	result;
    Vector<String>	lines;
    int			i;
    Enumeration<String>	enm;
    String		name;

    try {
      result = new StringBuilder();

      // create nested structure
      lines = nestedToLines(getOutput());

      // add meta-data
      if (m_OutputProlog) {
	result.append(COMMENT + " Project: " + Environment.getInstance().getProject() + "\n");
	result.append(COMMENT + " Date: " + m_DateFormat.format(new Date()) + "\n");
	result.append(COMMENT + " User: " + System.getProperty("user.name") + "\n");
        if (m_OutputClasspath)
	  result.append(COMMENT + " Class-Path: " + Java.getClassPath(true) + "\n");
	enm = Revisions.getSingleton().names();
	while (enm.hasMoreElements()) {
	  name = enm.nextElement();
	  result.append(COMMENT + " Revision/" + name + ": " + Revisions.getSingleton().get(name) + "\n");
	}
	result.append(COMMENT + "\n");
      }

      // add actual data
      for (i = 0; i < lines.size(); i++) {
	result.append(lines.get(i));
	result.append("\n");
      }
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    if (result == null)
      return null;
    else
      return result.toString();
  }

  /**
   * Hook-method before starting visiting options.
   */
  protected void preProduce() {
    Vector	nested;

    super.preProduce();

    m_Output.clear();
    m_Output.add(m_Input.getClass().getName());
    nested = new Vector();
    m_Output.add(nested);

    m_Nesting.push(nested);
  }

  /**
   * Visits the option and obtains information from it.
   *
   * @param option	the current option
   * @return		the last internal data structure that was generated
   */
  public Vector doProduce(AbstractOption option) {
    int		i;

    for (i = 0; i < m_Blacklisted.length; i++) {
      if (option.getReadMethod().getReturnType() == m_Blacklisted[i])
	return null;
    }

    return super.doProduce(option);
  }

  /**
   * Hook-method after visiting options.
   */
  protected void postProduce() {
    super.postProduce();

    m_Nesting.pop();
  }
}
