/*
 * Function.java
 * Copyright (C) 2008 University of Waikato, Hamilton, New Zealand
 */

package adams.data.fit;

import adams.core.ClassLister;
import adams.core.ConsoleObject;
import adams.core.option.OptionHandler;
import adams.core.option.OptionManager;
import adams.core.option.OptionUtils;

/**
 * Abstract superclass for arbitrary functions.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3136 $
 */
public abstract class Function
  extends ConsoleObject
  implements OptionHandler {

  /** for serialization. */
  private static final long serialVersionUID = -1400960650078696595L;

  /** for managing the available options. */
  protected OptionManager m_OptionManager;

  /** whether debugging is on. */
  protected boolean m_Debug;

  /**
   * Initializes the object, sets default options.
   */
  public Function() {
    super();
    initialize();
    defineOptions();
    getOptionManager().setDefaults();
  }

  /**
   * Returns a short description of the function.
   *
   * @return		a description of the function
   */
  public abstract String globalInfo();

  /**
   * Initializes the members.
   */
  protected void initialize() {
  }

  /**
   * Returns a new instance of the option manager.
   *
   * @return		the manager to use
   */
  protected OptionManager newOptionManager() {
    return new OptionManager(this);
  }

  /**
   * Adds options to the internal list of options. Derived classes must
   * override this method to add additional options.
   */
  public void defineOptions() {
    m_OptionManager = newOptionManager();

    m_OptionManager.add(
	"D", "debug", false);
  }

  /**
   * Returns the option manager.
   *
   * @return		the manager
   */
  public OptionManager getOptionManager() {
    if (m_OptionManager == null)
      defineOptions();

    return m_OptionManager;
  }

  /**
   * Cleans up the options.
   */
  public void cleanUpOptions() {
    if (m_OptionManager != null) {
      m_OptionManager.cleanUp();
      m_OptionManager = null;
    }
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Cleans up the options.
   *
   * @see	#cleanUpOptions()
   */
  public void destroy() {
    cleanUpOptions();
  }

  /**
   * Set debugging mode.
   *
   * @param value 	true if debug output should be printed
   */
  public void setDebug(boolean value) {
    m_Debug = value;
    getDebugging().setEnabled(value);
  }

  /**
   * Returns whether debugging is turned on.
   *
   * @return 		true if debugging output is on
   */
  public boolean getDebug() {
    return m_Debug;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String debugTipText() {
    return "If set to true, function may output additional info to the console.";
  }

  /**
   * Calculates the y value, based on the parameters and x value.
   *
   * @param x		the x value
   * @param a		the coefficients
   * @return		the y value
   */
  public abstract double calcY(double x, double[] a);

  /**
   * Returns the setup/commandline of the function.
   *
   * @return		the commandline
   */
  public String toString() {
    return OptionUtils.getCommandLine(this);
  }

  /**
   * Returns a list with classnames of functions.
   *
   * @return		the function classnames
   */
  public static String[] getFunctions() {
    return ClassLister.getSingleton().getClassnames(Function.class);
  }

  /**
   * Instantiates the function with the given options.
   *
   * @param classname	the classname of the function to instantiate
   * @param options	the options for the function
   * @return		the instantiated function or null if an error occurred
   */
  public static Function forName(String classname, String[] options) {
    Function	result;

    try {
      result = (Function) OptionUtils.forName(Function.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }
}
