/*
 * StringJoin.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.core.Utils;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Creates a single string out of an array of strings.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * <pre>   java.lang.String[]</pre>
 * - generates:<br/>
 * <pre>   java.lang.String</pre>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: StringJoin
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 *         The annotations to attach to this actor.
 *         default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-glue &lt;java.lang.String&gt; (property: glue)
 *         The string to use for 'glueing' the array elements together; \t\n\r\b\f
 *         get automatically converted into their character counterparts.
 *         default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2414 $
 */
public class StringJoin
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -5524254075032084295L;

  /** the string to use for glueing the strings together. */
  protected String m_Glue;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Creates a single string out of an array of strings.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "glue", "glue",
	    "");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("glue");

    if (variable != null)
      return variable;
    else if ((m_Glue != null) && (m_Glue.length() > 0))
      return Utils.backQuoteChars(m_Glue);
    else
      return null;
  }

  /**
   * Sets the string to use as 'glue' between the array elements.
   *
   * @param value	the string
   */
  public void setGlue(String value) {
    m_Glue = Utils.unbackQuoteChars(value);
    reset();
  }

  /**
   * Returns the string used as 'glue' between the array elements.
   *
   * @return		the string
   */
  public String getGlue() {
    return Utils.backQuoteChars(m_Glue);
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String glueTipText() {
    return
        "The string to use for 'glueing' the array elements together; \\t\\n\\r\\b\\f get "
      + "automatically converted into their character counterparts.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String[].class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String[].class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    StringBuilder	str;
    String[]		array;
    int			i;

    result = null;

    try {
      array = (String[]) m_InputToken.getPayload();
      str = new StringBuilder();
      for (i = 0; i < array.length; i++) {
	if (i > 0)
	  str.append(m_Glue);
	str.append(array[i]);
      }
      m_OutputToken = new Token(str.toString());
    }
    catch (Exception e) {
      m_OutputToken = null;
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
