/**
 * SpreadSheetTable.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.core;

import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Cell;

/**
 * A specialized table for displaying a SpreadSheet table model.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2314 $
 */
public class SpreadSheetTable
  extends SortableAndSearchableTable {

  /** for serialization. */
  private static final long serialVersionUID = 1333317577811620786L;

  /**
   * Initializes the table.
   *
     * @param sheet	the underlying spread sheet
   */
  public SpreadSheetTable(SpreadSheet sheet) {
    this(new SpreadSheetTableModel(sheet));
  }

  /**
   * Initializes the table.
   *
     * @param model	the underlying spread sheet model
   */
  public SpreadSheetTable(SpreadSheetTableModel model) {
    super(model);
  }

  /**
   * Sets the custom cell renderer.
   */
  protected void setCustomCellRenderer() {
    int				i;
    SpreadSheetCellRenderer	renderer;
    TableColumnModel		colModel;

    renderer = new SpreadSheetCellRenderer();
    colModel = getColumnModel();
    for (i = 1; i < getColumnCount(); i++)
      colModel.getColumn(i).setCellRenderer(renderer);
  }

  /**
   * Sets the model to display - only {@link #getTableModelClass()}.
   *
   * @param model	the model to display
   */
  public void setModel(TableModel model) {
    super.setModel(model);
    setCustomCellRenderer();
  }

  /**
   * Returns the initial setting of whether to set optimal column widths.
   *
   * @return		true
   */
  protected boolean initialUseOptimalColumnWidths() {
    return true;
  }

  /**
   * Returns the initial setting of whether to sort new models.
   *
   * @return		true
   */
  protected boolean initialSortNewTableModel() {
    return false;
  }

  /**
   * Returns the class of the table model that the models need to be derived
   * from. The default implementation just returns TableModel.class
   *
   * @return		the class the models must be derived from
   */
  protected Class getTableModelClass() {
    return SpreadSheetTableModel.class;
  }

  /**
   * Creates an empty default model.
   *
   * @return		the model
   */
  protected TableModel createDefaultDataModel() {
    return new SpreadSheetTableModel();
  }

  /**
   * Returns the spread sheet cell at the specified location.
   *
   * @param rowIndex	the current display row index
   * @param columnIndex	the column index
   * @return		the cell or null if invalid coordinates
   */
  public Cell getCellAt(int rowIndex, int columnIndex) {
    Cell			result;
    SpreadSheetTableModel	sheetmodel;

    sheetmodel = (SpreadSheetTableModel) getUnsortedModel();
    result     = sheetmodel.getCellAt(getActualRow(rowIndex), columnIndex);

    return result;
  }

  /**
   * Sets the number of decimals to display. Use -1 to display all.
   *
   * @param value	the number of decimals
   */
  public void setNumDecimals(int value) {
    ((SpreadSheetTableModel) getUnsortedModel()).setNumDecimals(value);
  }

  /**
   * Returns the currently set number of decimals. -1 if displaying all.
   *
   * @return		the number of decimals
   */
  public int getNumDecimals() {
    return ((SpreadSheetTableModel) getUnsortedModel()).getNumDecimals();
  }

  /**
   * Returns the underlying sheet.
   *
   * @return		the spread sheet
   */
  public SpreadSheet getSheet() {
    return ((SpreadSheetTableModel) getUnsortedModel()).getSheet();
  }
}