/**
 * AbstractPreferencesDialog.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.dialog;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Frame;
import java.util.Vector;

import adams.gui.chooser.AbstractChooserPanel;
import adams.gui.core.BaseDialog;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.ParameterPanel;

/**
 * Ancestor for dialogs that allow the user to set preferences.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4095 $
 */
public abstract class AbstractPreferencesDialog
  extends AbstractApprovalDialog {

  /** for serialization. */
  private static final long serialVersionUID = 7147933476003801800L;

  /** the panel for the preferences. */
  protected ParameterPanel m_PanelPreferences;

  /** the identifiers of the preference. */
  protected Vector<String> m_Identifiers;

  /**
   * Creates a modal preferences dialog without a title with the specified
   * Dialog as its owner.
   *
   * @param owner	the owning dialog
   */
  public AbstractPreferencesDialog(Dialog owner) {
    super(owner, ModalityType.DOCUMENT_MODAL);
  }

  /**
   * Creates a modal preferences dialog without a title with the specified
   * Frame as its owner.
   *
   * @param owner	the owning frame
   */
  public AbstractPreferencesDialog(Frame owner) {
    super(owner, true);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Identifiers = new Vector<String>();
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    super.initGUI();

    setTitle("Preferences");
    setDefaultCloseOperation(BaseDialog.DISPOSE_ON_CLOSE);

    // preferences
    m_PanelPreferences = new ParameterPanel();
    getContentPane().add(new BaseScrollPane(m_PanelPreferences), BorderLayout.CENTER);
  }

  /**
   * Removes all preferences.
   */
  protected void clearPreferences() {
    m_Identifiers.clear();
    m_PanelPreferences.clearParameters();
  }

  /**
   * Adds a preference.
   *
   * @param identifier	the unique identifier of the preference
   * @param label	the label to add, the mnemonic to use is preceded by "_"
   * @param comp	the component to add
   * @throws IllegalArgumentException	if the identifier already exists
   */
  public void addPreference(String identifier, String label, Component comp) {
    if (m_Identifiers.contains(identifier))
      throw new IllegalArgumentException("Identifier '" + identifier + "' already present!");
    m_Identifiers.add(identifier);
    m_PanelPreferences.addParameter(label, comp);
  }

  /**
   * Adds the chooser panel at the end.
   *
   * @param identifier	the unique identifier of the preference
   * @param label	the label to add, the mnemonic to use is preceded by "_"
   * @param chooser	the chooser panel to add
   * @throws IllegalArgumentException	if the identifier already exists
   */
  public void addPreference(String identifier, String label, AbstractChooserPanel chooser) {
    if (m_Identifiers.contains(identifier))
      throw new IllegalArgumentException("Identifier '" + identifier + "' already present!");
    m_Identifiers.add(identifier);
    m_PanelPreferences.addParameter(label, chooser);
  }

  /**
   * Returns the component at the specified location.
   *
   * @param index	the index of the specified location
   * @return		the component at the position
   */
  public Component getPreference(int index) {
    return m_PanelPreferences.getParameter(index);
  }

  /**
   * Returns the component associated with the identifier.
   *
   * @param identifier	the identifier of the preference to return
   * @return		the associated component, null if none found
   */
  public Component getPreference(String identifier) {
    int		index;

    index = m_Identifiers.indexOf(identifier);
    if (index == -1)
      return null;
    else
      return m_PanelPreferences.getParameter(index);
  }

  /**
   * Returns the number of preferences currently displayed.
   *
   * @return		the number of preferences
   */
  public int getPreferenceCount() {
    return m_PanelPreferences.getParameterCount();
  }
}
