/*
 * FlowEditorPanel.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.flow;

import java.awt.BorderLayout;
import java.awt.Dialog.ModalityType;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.Vector;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JSplitPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.ClassLister;
import adams.core.Pausable;
import adams.core.Properties;
import adams.core.StatusMessageHandler;
import adams.core.io.FilenameProposer;
import adams.core.io.PlaceholderFile;
import adams.data.statistics.InformativeStatistic;
import adams.db.LogEntryHandler;
import adams.env.Environment;
import adams.env.FlowEditorPanelDefinition;
import adams.env.FlowEditorPanelShortcutsDefinition;
import adams.flow.control.Flow;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorStatistic;
import adams.flow.core.ActorUtils;
import adams.flow.core.InstantiatableActor;
import adams.flow.processor.AbstractActorProcessor;
import adams.flow.processor.GraphicalOutputProducingProcessor;
import adams.flow.processor.ModifyingProcessor;
import adams.flow.processor.RemoveDisabledActors;
import adams.gui.action.AbstractBaseAction;
import adams.gui.action.ToggleAction;
import adams.gui.application.ChildFrame;
import adams.gui.application.ChildWindow;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.BaseDialog;
import adams.gui.core.BaseStatusBar;
import adams.gui.core.BaseStatusBar.StatusProcessor;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.core.RecentFilesHandler;
import adams.gui.core.ToolBarPanel;
import adams.gui.dialog.PropertiesPreferencesDialog;
import adams.gui.dialog.PropertiesPreferencesDialog.PreferenceType;
import adams.gui.event.RecentFileEvent;
import adams.gui.event.RecentFileListener;
import adams.gui.event.TabVisibilityChangeEvent;
import adams.gui.event.TabVisibilityChangeListener;
import adams.gui.event.UndoEvent;
import adams.gui.flow.tab.FlowTabManager;
import adams.gui.flow.tree.Tree;
import adams.gui.goe.GenericObjectEditorDialog;
import adams.gui.sendto.SendToActionSupporter;
import adams.gui.sendto.SendToActionUtils;
import adams.gui.tools.VariableManagementPanel;
import adams.gui.visualization.statistics.InformativeStatisticFactory;

/**
 * A panel for setting up, modifying, saving and loading "simple" flows.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4168 $
 */
public class FlowEditorPanel
  extends ToolBarPanel
  implements MenuBarProvider, StatusMessageHandler, SendToActionSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -3579084888256133873L;

  /** the name of the props file with the general properties. */
  public final static String FILENAME = "FlowEditor.props";

  /** the name of the props file with the shortcuts. */
  public final static String FILENAME_SHORTCUTS = "FlowEditorShortcuts.props";

  /** the file to store the recent files in. */
  public final static String SESSION_FILE = "FlowSession.props";

  /** the default title for dialogs/frames. */
  public final static String DEFAULT_TITLE = "Flow editor";

  /** the general properties. */
  protected static Properties m_Properties;

  /** the shortcut properties. */
  protected static Properties m_PropertiesShortcuts;

  /** the panel itself. */
  protected FlowEditorPanel m_Self;

  /** the menu bar, if used. */
  protected JMenuBar m_MenuBar;

  /** the "new" sub-menu. */
  protected JMenu m_MenuFileNew;

  /** the "load" action. */
  protected AbstractBaseAction m_ActionFileLoad;

  /** the "load recent" submenu. */
  protected JMenu m_MenuFileLoadRecent;

  /** the "new" action. */
  protected AbstractBaseAction m_ActionFileNew;

  /** the "save" action. */
  protected AbstractBaseAction m_ActionFileSave;

  /** the "save as" action. */
  protected AbstractBaseAction m_ActionFileSaveAs;

  /** the "revert" action. */
  protected AbstractBaseAction m_ActionFileRevert;

  /** the "export" action. */
  protected AbstractBaseAction m_ActionFileExport;

  /** the "import" action. */
  protected AbstractBaseAction m_ActionFileImport;

  /** the "preferences" action. */
  protected AbstractBaseAction m_ActionFilePreferences;

  /** the "close tab" action. */
  protected AbstractBaseAction m_ActionFileCloseTab;

  /** the "close" action. */
  protected AbstractBaseAction m_ActionFileClose;

  /** the toggle undo action. */
  protected AbstractBaseAction m_ActionEditEnableUndo;

  /** the undo action. */
  protected AbstractBaseAction m_ActionEditUndo;

  /** the redo action. */
  protected AbstractBaseAction m_ActionEditRedo;

  /** the find action. */
  protected AbstractBaseAction m_ActionEditFind;

  /** the find next action. */
  protected AbstractBaseAction m_ActionEditFindNext;

  /** the locate actor action. */
  protected AbstractBaseAction m_ActionEditLocateActor;

  /** the remove disabled actors action. */
  protected AbstractBaseAction m_ActionEditCleanUpFlow;

  /** the "process actors" action. */
  protected AbstractBaseAction m_ActionEditProcessActors;

  /** the "enable all breakpoints" action. */
  protected AbstractBaseAction m_ActionDebugEnableAllBreakpoints;

  /** the "disable all breakpoints" action. */
  protected AbstractBaseAction m_ActionDebugDisableAllBreakpoints;

  /** the "variables" action. */
  protected AbstractBaseAction m_ActionDebugVariables;

  /** the "headless" action. */
  protected AbstractBaseAction m_ActionExecutionHeadless;

  /** the "check setup" action. */
  protected AbstractBaseAction m_ActionExecutionValidateSetup;

  /** the "run" action. */
  protected AbstractBaseAction m_ActionExecutionRun;

  /** the "pause" action. */
  protected AbstractBaseAction m_ActionExecutionPauseAndResume;

  /** the "stop" action. */
  protected AbstractBaseAction m_ActionExecutionStop;

  /** the "display errors" action. */
  protected AbstractBaseAction m_ActionExecutionDisplayErrors;

  /** the "Clear graphical output" action. */
  protected AbstractBaseAction m_ActionExecutionClearGraphicalOutput;

  /** the "show toolbar" action. */
  protected AbstractBaseAction m_ActionViewShowToolbar;

  /** the "show quick info" action. */
  protected AbstractBaseAction m_ActionViewShowQuickInfo;

  /** the "show annotations" action. */
  protected AbstractBaseAction m_ActionViewShowAnnotations;

  /** the "show input/output" action. */
  protected AbstractBaseAction m_ActionViewShowInputOutput;

  /** the highlight variables action. */
  protected AbstractBaseAction m_ActionViewHighlightVariables;

  /** the remove variable highlights action. */
  protected AbstractBaseAction m_ActionViewRemoveVariableHighlights;

  /** the "show source" action. */
  protected AbstractBaseAction m_ActionViewShowSource;

  /** the "statistic" action. */
  protected AbstractBaseAction m_ActionViewStatistics;

  /** the "new window" action. */
  protected AbstractBaseAction m_ActionNewWindow;

  /** the "duplicate tab in new window" action. */
  protected AbstractBaseAction m_ActionDuplicateTabInNewWindow;

  /** the "duplicate tab" action. */
  protected AbstractBaseAction m_ActionDuplicateTab;

  /** the filedialog for loading/saving flows. */
  protected BaseFileChooser m_FileChooser;

  /** the status. */
  protected BaseStatusBar m_StatusBar;

  /** the recent files handler. */
  protected RecentFilesHandler m_RecentFilesHandler;

  /** for proposing filenames for new flows. */
  protected FilenameProposer m_FilenameProposer;

  /** the split pane for displaying flow and tabs. */
  protected JSplitPane m_SplitPane;

  /** the tabbedpane for the flow panels. */
  protected FlowTabbedPane m_FlowPanels;

  /** the tabbedpane for the tabs. */
  protected FlowTabManager m_Tabs;

  /** the last variable search performed. */
  protected String m_LastVariableSearch;

  /** the dialog for importing the flow. */
  protected ImportDialog m_ImportDialog;

  /** the dialog for exporting the flow. */
  protected ExportDialog m_ExportDialog;

  /** the dialog for processing actors. */
  protected GenericObjectEditorDialog m_DialogProcessActors;

  /** the default toolbar location to use. */
  protected ToolBarLocation m_ToolBarLocation;

  /** the panel with the variables. */
  protected VariableManagementPanel m_PanelVariables;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Self                = this;
    m_RecentFilesHandler  = null;
    m_LastVariableSearch  = "";
    m_FileChooser         = new BaseFileChooser();
    m_FileChooser.addChoosableFileFilter(new ExtensionFileFilter("Flow setups", AbstractActor.FILE_EXTENSION));
    m_FileChooser.setCurrentDirectory(new PlaceholderFile(getProperties().getString("InitialDir", "%h")));
    m_FileChooser.setDefaultExtension(AbstractActor.FILE_EXTENSION);
    m_FileChooser.setAutoAppendExtension(true);
    m_FileChooser.setMultiSelectionEnabled(true);
    m_FilenameProposer    = new FilenameProposer(FlowPanel.PREFIX_NEW, AbstractActor.FILE_EXTENSION, getProperties().getString("InitialDir", "%h"));
    m_ExportDialog        = null;
    m_DialogProcessActors = null;
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    Properties			props;

    super.initGUI();

    props = getProperties();

    getContentPanel().setLayout(new BorderLayout());

    m_ToolBarLocation = ToolBarLocation.valueOf(props.getProperty("ToolBar.Location", "NORTH"));
    if (m_ToolBarLocation == ToolBarLocation.HIDDEN)
      m_ToolBarLocation = ToolBarLocation.NORTH;
    setToolBarLocation(ToolBarLocation.valueOf(props.getProperty("ToolBar.Location", "NORTH")));

    m_SplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, true);
    m_SplitPane.setDividerLocation(props.getInteger("DividerLocation", 500));
    m_SplitPane.setOneTouchExpandable(true);
    m_SplitPane.setResizeWeight(0.5);
    getContentPanel().add(m_SplitPane, BorderLayout.CENTER);

    // the flows
    m_FlowPanels = new FlowTabbedPane(this);
    m_SplitPane.setLeftComponent(m_FlowPanels);

    // the tabs
    m_Tabs = new FlowTabManager();
    m_Tabs.addTabVisibilityChangeListener(new TabVisibilityChangeListener() {
      public void tabVisibilityChanged(TabVisibilityChangeEvent e) {
	if (m_Tabs.getTabCount() == 0)
	  m_SplitPane.setRightComponent(null);
	else
	  m_SplitPane.setRightComponent(m_Tabs);
      }
    });
    if (m_Tabs.getTabCount() > 0)
      m_SplitPane.setRightComponent(m_Tabs);

    // the status
    m_StatusBar = new BaseStatusBar();
    m_StatusBar.setDialogSize(new Dimension(props.getInteger("StatusBar.Width", 600), props.getInteger("StatusBar.Height", 400)));
    m_StatusBar.setMouseListenerActive(true);
    m_StatusBar.setStatusProcessor(new StatusProcessor() {
      public String process(String msg) {
        return msg.replace(": ", ":\n");
      }
    });
    getContentPanel().add(m_StatusBar, BorderLayout.SOUTH);
  }

  /**
   * Initializes the actions.
   */
  @SuppressWarnings("serial")
  protected void initActions() {
    AbstractBaseAction	action;
    Properties		props;

    props = getProperties();

    // File/New (flow)
    action = new AbstractBaseAction("Flow", "new.gif") {
      public void actionPerformed(ActionEvent e) {
	newTab();
      }
    };
    action.setMnemonic(KeyEvent.VK_N);
    action.setAccelerator(getEditorShortcut("File.New"));
    m_ActionFileNew = action;

    // File/Open
    action = new AbstractBaseAction("Open...", "open.gif") {
      public void actionPerformed(ActionEvent e) {
	open();
      }
    };
    action.setMnemonic(KeyEvent.VK_O);
    action.setAccelerator(getEditorShortcut("File.Open"));
    m_ActionFileLoad = action;

    // File/Save
    action = new AbstractBaseAction("Save", "save.gif") {
      public void actionPerformed(ActionEvent e) {
	save();
      }
    };
    action.setMnemonic(KeyEvent.VK_S);
    action.setAccelerator(getEditorShortcut("File.Save"));
    m_ActionFileSave = action;

    // File/Save
    action = new AbstractBaseAction("Save as...", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	saveAs();
      }
    };
    action.setMnemonic(KeyEvent.VK_A);
    action.setAccelerator(getEditorShortcut("File.SaveAs"));
    m_ActionFileSaveAs = action;

    // File/Revert
    action = new AbstractBaseAction("Revert", "refresh.gif") {
      public void actionPerformed(ActionEvent e) {
	revert();
      }
    };
    action.setMnemonic(KeyEvent.VK_R);
    action.setAccelerator(getEditorShortcut("File.Revert"));
    m_ActionFileRevert = action;

    // File/Close tab
    action = new AbstractBaseAction("Close tab") {
      public void actionPerformed(ActionEvent e) {
	m_FlowPanels.removeSelectedTab();
      }
    };
    action.setMnemonic(KeyEvent.VK_T);
    action.setAccelerator(getEditorShortcut("File.CloseTab"));
    m_ActionFileCloseTab = action;

    // File/Import
    action = new AbstractBaseAction("Import...", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	importFlow();
      }
    };
    action.setMnemonic(KeyEvent.VK_I);
    action.setAccelerator(getEditorShortcut("File.Import"));
    m_ActionFileImport = action;

    // File/Export
    action = new AbstractBaseAction("Export...", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	exportFlow();
      }
    };
    action.setMnemonic(KeyEvent.VK_E);
    action.setAccelerator(getEditorShortcut("File.Export"));
    m_ActionFileExport = action;

    // File/Preferences
    action = new AbstractBaseAction("Preferences...", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	preferences();
      }
    };
    action.setMnemonic(KeyEvent.VK_P);
    action.setAccelerator(getEditorShortcut("File.Preferences"));
    m_ActionFilePreferences = action;

    // File/Close
    action = new AbstractBaseAction("Close", "exit.png") {
      public void actionPerformed(ActionEvent e) {
	close();
      }
    };
    action.setMnemonic(KeyEvent.VK_C);
    action.setAccelerator(getEditorShortcut("File.Exit"));
    m_ActionFileClose = action;

    // Edit/Enable Undo
    action = new AbstractBaseAction("Undo enabled", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	if (hasCurrentPanel())
	  getCurrentPanel().getUndo().setEnabled(!getCurrentPanel().getUndo().isEnabled());
      }
    };
    action.setMnemonic(KeyEvent.VK_N);
    action.setAccelerator(getEditorShortcut("Edit.ToggleUndo"));
    action.setEnabled(false);
    action.setSelected(true);
    m_ActionEditEnableUndo = action;

    // Edit/Undo
    action = new AbstractBaseAction("Undo", "undo.gif") {
      public void actionPerformed(ActionEvent e) {
	undo();
      }
    };
    action.setMnemonic(KeyEvent.VK_U);
    action.setAccelerator(getEditorShortcut("Edit.Undo"));
    action.setEnabled(false);
    m_ActionEditUndo = action;

    // Edit/Redo
    action = new AbstractBaseAction("Redo", "redo.gif") {
      public void actionPerformed(ActionEvent e) {
	redo();
      }
    };
    action.setMnemonic(KeyEvent.VK_R);
    action.setAccelerator(getEditorShortcut("Edit.Redo"));
    action.setEnabled(false);
    m_ActionEditRedo = action;

    // Edit/Find
    action = new AbstractBaseAction("Find", "find.gif") {
      public void actionPerformed(ActionEvent e) {
	find();
      }
    };
    action.setMnemonic(KeyEvent.VK_F);
    action.setAccelerator(getEditorShortcut("Edit.Find"));
    m_ActionEditFind = action;

    // Edit/Find next
    action = new AbstractBaseAction("Find next", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	findNext();
      }
    };
    action.setMnemonic(KeyEvent.VK_N);
    action.setAccelerator(getEditorShortcut("Edit.FindNext"));
    m_ActionEditFindNext = action;

    // Edit/Locate actor
    action = new AbstractBaseAction("Locate actor", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	locateActor();
      }
    };
    action.setMnemonic(KeyEvent.VK_L);
    action.setAccelerator(getEditorShortcut("Edit.LocateActor"));
    m_ActionEditLocateActor = action;

    // Edit/Remove disabled actors
    action = new AbstractBaseAction("Clean up flow", "delete.gif") {
      public void actionPerformed(ActionEvent e) {
	cleanUpFlow();
      }
    };
    action.setMnemonic(KeyEvent.VK_C);
    m_ActionEditCleanUpFlow = action;

    // Edit/Process actors
    action = new AbstractBaseAction("Process actors...", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	processActors();
      }
    };
    action.setMnemonic(KeyEvent.VK_D);
    action.setAccelerator(getEditorShortcut("Edit.ProcessActors"));
    m_ActionEditProcessActors = action;

    // Debug/Enable all breakpoints
    action = new AbstractBaseAction("Enable all breakpoints", "debug.png") {
      public void actionPerformed(ActionEvent e) {
	enableBreakpoints(true);
      }
    };
    action.setMnemonic(KeyEvent.VK_E);
    action.setAccelerator(getEditorShortcut("Debug.EnableAllBreakpoints"));
    m_ActionDebugEnableAllBreakpoints = action;

    // Debug/Disable all breakpoints
    action = new AbstractBaseAction("Disable all breakpoints", "debug_off.png") {
      public void actionPerformed(ActionEvent e) {
	enableBreakpoints(false);
      }
    };
    action.setMnemonic(KeyEvent.VK_D);
    action.setAccelerator(getEditorShortcut("Debug.DisableAllBreakpoints"));
    m_ActionDebugDisableAllBreakpoints = action;

    // Debug/Variables
    action = new AbstractBaseAction("Variables", "variable.gif") {
      public void actionPerformed(ActionEvent e) {
	showVariables();
      }
    };
    action.setMnemonic(KeyEvent.VK_V);
    action.setAccelerator(getEditorShortcut("Debug.Variables"));
    m_ActionDebugVariables = action;

    // Execution/Validate setup
    action = new AbstractBaseAction("Validate setup", "validate.png") {
      public void actionPerformed(ActionEvent e) {
	validateSetup();
      }
    };
    action.setMnemonic(KeyEvent.VK_V);
    action.setAccelerator(getEditorShortcut("Execution.ValidateSetup"));
    m_ActionExecutionValidateSetup = action;

    // Execution/Run
    action = new AbstractBaseAction("Run", "run.gif") {
      public void actionPerformed(ActionEvent e) {
	run();
      }
    };
    action.setMnemonic(KeyEvent.VK_R);
    action.setAccelerator(getEditorShortcut("Execution.Run"));
    m_ActionExecutionRun = action;

    // Execution/Pause+Resume
    action = new AbstractBaseAction("Pause", "pause.gif") {
      public void actionPerformed(ActionEvent e) {
	pauseAndResume();
      }
    };
    action.setMnemonic(KeyEvent.VK_U);
    action.setAccelerator(getEditorShortcut("Execution.PauseResume"));
    m_ActionExecutionPauseAndResume = action;

    // Execution/Stop
    action = new AbstractBaseAction("Stop", "stop_blue.gif") {
      public void actionPerformed(ActionEvent e) {
	stop();
      }
    };
    action.setMnemonic(KeyEvent.VK_S);
    action.setAccelerator(getEditorShortcut("Execution.Stop"));
    m_ActionExecutionStop = action;

    // Execution/Display errors
    action = new AbstractBaseAction("Display errors...", "log.gif") {
      public void actionPerformed(ActionEvent e) {
	displayErrors();
      }
    };
    action.setMnemonic(KeyEvent.VK_D);
    m_ActionExecutionDisplayErrors = action;

    // Execution/Clear graphical output
    action = new AbstractBaseAction("Clear graphical output", "close_window.png") {
      public void actionPerformed(ActionEvent e) {
	cleanUp();
	update();
      }
    };
    action.setMnemonic(KeyEvent.VK_C);
    action.setAccelerator(getEditorShortcut("Execution.ClearGraphicalOutput"));
    m_ActionExecutionClearGraphicalOutput = action;

    // Execution/Headless
    action = new ToggleAction("Headless", GUIHelper.getEmptyIcon());
    action.setMnemonic(KeyEvent.VK_H);
    action.setAccelerator(getEditorShortcut("Execution.ToggleHeadless"));
    m_ActionExecutionHeadless = action;

    // View/Show toolbar
    action = new AbstractBaseAction("Show toolbar", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	if (getToolBarLocation() == ToolBarLocation.HIDDEN)
	  setToolBarLocation(m_ToolBarLocation);
	else
	  setToolBarLocation(ToolBarLocation.HIDDEN);
      }
    };
    action.setMnemonic(KeyEvent.VK_T);
    action.setAccelerator(getEditorShortcut("View.ShowToolbar"));
    action.setSelected(getToolBarLocation() != ToolBarLocation.HIDDEN);
    m_ActionViewShowToolbar = action;

    // View/Show quick info
    action = new AbstractBaseAction("Show quick info", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	if (hasCurrentPanel())
	  getCurrentPanel().getTree().setShowQuickInfo(m_ActionViewShowQuickInfo.isSelected());
      }
    };
    action.setMnemonic(KeyEvent.VK_Q);
    action.setAccelerator(getEditorShortcut("View.ShowQuickInfo"));
    action.setSelected(props.getBoolean("ShowQuickInfo", true));
    m_ActionViewShowQuickInfo = action;

    // View/Show annotations
    action = new AbstractBaseAction("Show annotations", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	if (hasCurrentPanel())
	  getCurrentPanel().getTree().setShowAnnotations(m_ActionViewShowAnnotations.isSelected());
      }
    };
    action.setMnemonic(KeyEvent.VK_A);
    action.setAccelerator(getEditorShortcut("View.ShowAnnotations"));
    action.setSelected(props.getBoolean("ShowAnnotations", false));
    m_ActionViewShowAnnotations = action;

    // View/Show input/output info
    action = new AbstractBaseAction("Show input/output", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	if (hasCurrentPanel())
	  getCurrentPanel().getTree().setShowInputOutput(m_ActionViewShowInputOutput.isSelected());
      }
    };
    action.setMnemonic(KeyEvent.VK_P);
    action.setAccelerator(getEditorShortcut("View.ShowInputOutput"));
    action.setSelected(props.getBoolean("ShowInputOutput", false));
    m_ActionViewShowInputOutput = action;

    // View/Highlight variables
    action = new AbstractBaseAction("Highlight variables...", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	highlightVariables(true);
      }
    };
    action.setMnemonic(KeyEvent.VK_V);
    action.setAccelerator(getEditorShortcut("View.HighlightVariables"));
    m_ActionViewHighlightVariables = action;

    // View/Remove variable highlights
    action = new AbstractBaseAction("Remove variable highlights", GUIHelper.getEmptyIcon()) {
      public void actionPerformed(ActionEvent e) {
	highlightVariables(false);
      }
    };
    action.setMnemonic(KeyEvent.VK_R);
    action.setAccelerator(getEditorShortcut("View.RemoveVariableHighlights"));
    m_ActionViewRemoveVariableHighlights = action;

    // View/Show source
    action = new AbstractBaseAction("Show source...", "source.png") {
      public void actionPerformed(ActionEvent e) {
	showSource();
      }
    };
    action.setMnemonic(KeyEvent.VK_S);
    action.setAccelerator(getEditorShortcut("View.ShowSource"));
    m_ActionViewShowSource = action;

    // View/Statistics
    action = new AbstractBaseAction("Statistics...", "statistics.png") {
      public void actionPerformed(ActionEvent e) {
	statistics();
      }
    };
    action.setMnemonic(KeyEvent.VK_T);
    action.setAccelerator(getEditorShortcut("View.Statistics"));
    m_ActionViewStatistics = action;

    // Window/New Window
    action = new AbstractBaseAction("New window") {
      public void actionPerformed(ActionEvent e) {
	newWindow();
      }
    };
    action.setMnemonic(KeyEvent.VK_W);
    action.setAccelerator(getEditorShortcut("Window.NewWindow"));
    m_ActionNewWindow = action;

    // Window/Duplicate in new window
    action = new AbstractBaseAction("Duplicate in new window") {
      public void actionPerformed(ActionEvent e) {
	duplicateTabInNewWindow();
      }
    };
    action.setMnemonic(KeyEvent.VK_D);
    action.setAccelerator(getEditorShortcut("Window.DuplicateInNewWindow"));
    m_ActionDuplicateTabInNewWindow = action;

    // Window/Duplicate in new tab
    action = new AbstractBaseAction("Duplicate in new tab", "copy.gif") {
      public void actionPerformed(ActionEvent e) {
	duplicateTab();
      }
    };
    action.setMnemonic(KeyEvent.VK_D);
    action.setAccelerator(getEditorShortcut("Window.DuplicateInNewWindow"));
    m_ActionDuplicateTab = action;
  }

  /**
   * Initializes the toolbar.
   */
  protected void initToolBar() {
    addToToolBar(m_ActionFileNew);
    addToToolBar(m_ActionFileLoad);
    addToToolBar(m_ActionFileSave);
    addSeparator();
    addToToolBar(m_ActionEditUndo);
    addToToolBar(m_ActionEditRedo);
    addSeparator();
    addToToolBar(m_ActionEditFind);
    addSeparator();
    addToToolBar(m_ActionExecutionValidateSetup);
    addToToolBar(m_ActionExecutionRun);
    addToToolBar(m_ActionExecutionPauseAndResume);
    addToToolBar(m_ActionExecutionStop);
  }

  /**
   * Creates a menu bar (singleton per panel object). Can be used in frames.
   *
   * @return		the menu bar
   */
  public JMenuBar getMenuBar() {
    JMenuBar		result;
    JMenu		menu;
    JMenu		submenu;
    JMenuItem		menuitem;
    String[]		actors;
    int			i;

    if (m_MenuBar == null) {
      // register window listener since we're part of a dialog or frame
      if (getParentFrame() != null) {
	final JFrame frame = (JFrame) getParentFrame();
	frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
	frame.addWindowListener(new WindowAdapter() {
	  public void windowClosing(WindowEvent e) {
	    close();
	  }
	});
      }
      else if (getParentDialog() != null) {
	final JDialog dialog = (JDialog) getParentDialog();
	dialog.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
	dialog.addWindowListener(new WindowAdapter() {
	  public void windowClosing(WindowEvent e) {
	    close();
	  }
	});
      }

      result = new JMenuBar();

      // File
      menu = new JMenu("File");
      result.add(menu);
      menu.setMnemonic('F');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateActions();
	}
      });

      // File/New
      submenu = new JMenu("New");
      menu.add(submenu);
      submenu.setMnemonic('N');
      submenu.setIcon(GUIHelper.getIcon("new.gif"));
      m_MenuFileNew = submenu;
      actors = ClassLister.getSingleton().getClassnames(InstantiatableActor.class);
      for (i = 0; i < actors.length; i++) {
	final AbstractActor actor = AbstractActor.forName(actors[i], new String[0]);;
	if (actor instanceof Flow) {
	  menuitem = new JMenuItem(m_ActionFileNew);
	  submenu.add(menuitem);
	}
	else {
	  menuitem = new JMenuItem(actors[i].replaceAll(".*\\.", ""));
	  submenu.add(menuitem);
	  menuitem.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      newFlow(actor);
	    }
	  });
	}
      }

      menu.add(new JMenuItem(m_ActionFileLoad));

      // File/Recent files
      submenu = new JMenu("Open recent");
      menu.add(submenu);
      m_RecentFilesHandler = new RecentFilesHandler(
	  SESSION_FILE, getProperties().getInteger("MaxRecentFlows", 5), submenu);
      m_RecentFilesHandler.addRecentFileListener(new RecentFileListener() {
	public void recentFileAdded(RecentFileEvent e) {
	  // ignored
	}
	public void recentFileSelected(RecentFileEvent e) {
	  FlowPanel panel = m_FlowPanels.newPanel();
	  panel.load(e.getFile());
	}
      });
      m_MenuFileLoadRecent = submenu;

      menu.add(new JMenuItem(m_ActionFileSave));
      menu.add(new JMenuItem(m_ActionFileSaveAs));
      menu.add(new JMenuItem(m_ActionFileRevert));
      menu.add(new JMenuItem(m_ActionFileCloseTab));
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionFileImport));
      menu.add(new JMenuItem(m_ActionFileExport));
      SendToActionUtils.addSendToSubmenu(this, menu);
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionFilePreferences));
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionFileClose));

      // Edit
      menu = new JMenu("Edit");
      result.add(menu);
      menu.setMnemonic('E');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateActions();
	}
      });

      menu.add(new JCheckBoxMenuItem(m_ActionEditEnableUndo));
      menu.add(new JMenuItem(m_ActionEditUndo));
      menu.add(new JMenuItem(m_ActionEditRedo));
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionEditFind));
      menu.add(new JMenuItem(m_ActionEditFindNext));
      menu.add(new JMenuItem(m_ActionEditLocateActor));
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionEditCleanUpFlow));
      menu.add(new JMenuItem(m_ActionEditProcessActors));

      // Debug
      menu = new JMenu("Debug");
      result.add(menu);
      menu.setMnemonic('D');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateActions();
	}
      });

      menu.add(new JMenuItem(m_ActionDebugEnableAllBreakpoints));
      menu.add(new JMenuItem(m_ActionDebugDisableAllBreakpoints));
      menu.add(new JMenuItem(m_ActionDebugVariables));

      // Execution
      menu = new JMenu("Execution");
      result.add(menu);
      menu.setMnemonic('E');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateActions();
	}
      });

      menu.add(new JMenuItem(m_ActionExecutionValidateSetup));
      menu.add(new JMenuItem(m_ActionExecutionRun));
      menu.add(new JMenuItem(m_ActionExecutionPauseAndResume));
      menu.add(new JMenuItem(m_ActionExecutionStop));
      menu.add(new JMenuItem(m_ActionExecutionDisplayErrors));
      menu.add(new JMenuItem(m_ActionExecutionClearGraphicalOutput));
      menu.addSeparator();
      menu.add(new JCheckBoxMenuItem(m_ActionExecutionHeadless));

      // View
      menu = new JMenu("View");
      result.add(menu);
      menu.setMnemonic('V');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateActions();
	}
      });

      menu.add(new JCheckBoxMenuItem(m_ActionViewShowToolbar));
      menu.add(new JCheckBoxMenuItem(m_ActionViewShowQuickInfo));
      menu.add(new JCheckBoxMenuItem(m_ActionViewShowAnnotations));
      menu.add(new JCheckBoxMenuItem(m_ActionViewShowInputOutput));
      m_Tabs.addTabsSubmenu(menu);
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionViewHighlightVariables));
      menu.add(new JMenuItem(m_ActionViewRemoveVariableHighlights));
      menu.addSeparator();
      menu.add(new JMenuItem(m_ActionViewShowSource));
      menu.add(new JMenuItem(m_ActionViewStatistics));

      // Window
      if ((GUIHelper.getParent(m_Self, ChildFrame.class) != null) && (getParentDialog() == null)) {
	menu = new JMenu("Window");
	result.add(menu);
	menu.setMnemonic('W');
	menu.addChangeListener(new ChangeListener() {
	  public void stateChanged(ChangeEvent e) {
	    updateActions();
	  }
	});

	menu.add(new JMenuItem(m_ActionNewWindow));
	menu.add(new JMenuItem(m_ActionDuplicateTabInNewWindow));
	menu.add(new JMenuItem(m_ActionDuplicateTab));
      }

      // update menu
      m_MenuBar = result;
      updateActions();
    }
    else {
      result = m_MenuBar;
    }

    return result;
  }

  /**
   * Returns the tab manager.
   *
   * @return		the tabs
   */
  public FlowTabManager getTabs() {
    return m_Tabs;
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateActions() {
    boolean	inputEnabled;
    Pausable	pausable;
    boolean	hasCurrent;

    hasCurrent = hasCurrentPanel();
    if (hasCurrent)
      getCurrentPanel().updateTitle();

    if (m_MenuBar == null)
      return;

    inputEnabled = !isRunning() && !isStopping() && !isSwingWorkerRunning();

    if ((getCurrentFlow() != null) && (getCurrentFlow() instanceof Pausable))
      pausable = (Pausable) getCurrentFlow();
    else
      pausable = null;

    // File
    m_MenuFileNew.setEnabled(true);
    m_ActionFileNew.setEnabled(true);
    m_ActionFileLoad.setEnabled(true);
    m_MenuFileLoadRecent.setEnabled(m_RecentFilesHandler.size() > 0);
    m_ActionFileSave.setEnabled(inputEnabled && hasCurrent);
    m_ActionFileSaveAs.setEnabled(inputEnabled && hasCurrent);
    m_ActionFileImport.setEnabled(true);
    m_ActionFileExport.setEnabled(inputEnabled && hasCurrent);
    m_ActionFileRevert.setEnabled(inputEnabled && hasCurrent && (getCurrentPanel().getCurrentFile() != null) && getCurrentPanel().getTree().isModified());
    m_ActionFileCloseTab.setEnabled(hasCurrent);
    m_ActionFileClose.setEnabled(!isAnyRunning() && !isAnyStopping() && !isAnySwingWorkerRunning());

    // Edit
    m_ActionEditEnableUndo.setEnabled(inputEnabled && hasCurrent);
    m_ActionEditEnableUndo.setSelected(hasCurrent && getCurrentPanel().getUndo().isEnabled());
    m_ActionEditUndo.setEnabled(inputEnabled && hasCurrent && getCurrentPanel().getUndo().canUndo());
    if (hasCurrent && getCurrentPanel().getUndo().canUndo()) {
      m_ActionEditUndo.setName("Undo - " + getCurrentPanel().getUndo().peekUndoComment(true));
      m_ActionEditUndo.setToolTipText(getCurrentPanel().getUndo().peekUndoComment());
    }
    else {
      m_ActionEditUndo.setName("Undo");
      m_ActionEditUndo.setToolTipText(null);
    }
    m_ActionEditRedo.setEnabled(inputEnabled && hasCurrent && getCurrentPanel().getUndo().canRedo());
    if (hasCurrent && getCurrentPanel().getUndo().canRedo()) {
      m_ActionEditRedo.setName("Redo - " + getCurrentPanel().getUndo().peekRedoComment(true));
      m_ActionEditRedo.setToolTipText(getCurrentPanel().getUndo().peekRedoComment());
    }
    else {
      m_ActionEditRedo.setName("Redo");
      m_ActionEditRedo.setToolTipText(null);
    }
    m_ActionEditCleanUpFlow.setEnabled(inputEnabled);
    m_ActionEditProcessActors.setEnabled(inputEnabled);
    m_ActionEditFind.setEnabled(true);
    m_ActionEditFindNext.setEnabled(hasCurrent && (getCurrentPanel().getTree().getLastSearchNode() != null));
    m_ActionEditLocateActor.setEnabled(true);
    m_ActionViewHighlightVariables.setEnabled(true);
    m_ActionViewRemoveVariableHighlights.setEnabled(true);

    // Debug
    m_ActionDebugEnableAllBreakpoints.setEnabled(inputEnabled);
    m_ActionDebugDisableAllBreakpoints.setEnabled(inputEnabled);
    m_ActionDebugVariables.setEnabled(isRunning());

    // Execution
    m_ActionExecutionValidateSetup.setEnabled(inputEnabled && hasCurrent);
    m_ActionExecutionRun.setEnabled(inputEnabled && hasCurrent && getCurrentPanel().getTree().isFlow());
    if ((pausable != null) && pausable.isPaused()) {
      m_ActionExecutionPauseAndResume.setIcon(GUIHelper.getIcon("resume.gif"));
      m_ActionExecutionPauseAndResume.setName("Resume");
    }
    else {
      m_ActionExecutionPauseAndResume.setIcon(GUIHelper.getIcon("pause.gif"));
      m_ActionExecutionPauseAndResume.setName("Pause");
    }
    m_ActionExecutionPauseAndResume.setEnabled(isRunning());
    m_ActionExecutionStop.setEnabled(isRunning());
    m_ActionExecutionHeadless.setEnabled(inputEnabled);
    m_ActionExecutionDisplayErrors.setEnabled(
	inputEnabled && (getLastFlow() != null)
	&& (getLastFlow() instanceof LogEntryHandler)
	&& (((LogEntryHandler) getLastFlow()).countLogEntries() > 0));
    m_ActionExecutionClearGraphicalOutput.setEnabled(inputEnabled && (getLastFlow() != null));

    // View
    m_ActionViewShowQuickInfo.setEnabled(hasCurrent);
    m_ActionViewShowAnnotations.setEnabled(hasCurrent);
    m_ActionViewShowInputOutput.setEnabled(hasCurrent);
    m_ActionViewStatistics.setEnabled(hasCurrent);
    m_ActionViewShowSource.setEnabled(hasCurrent);
    m_ActionViewHighlightVariables.setEnabled(hasCurrent);
    m_ActionViewRemoveVariableHighlights.setEnabled(hasCurrent);

    // Window
    m_ActionNewWindow.setEnabled(true);
    m_ActionDuplicateTabInNewWindow.setEnabled(hasCurrent);
    m_ActionDuplicateTab.setEnabled(hasCurrent);
  }

  /**
   * Updates the enabled state of the widgets.
   */
  protected void updateWidgets() {
    boolean	inputEnabled;

    inputEnabled = !isRunning() && !isStopping();

    if (hasCurrentPanel())
      getCurrentPanel().getTree().setEditable(inputEnabled);
  }

  /**
   * updates the enabled state etc. of all the GUI elements.
   */
  public void update() {
    updateActions();
    updateWidgets();
    if (hasCurrentPanel())
      getCurrentPanel().updateTitle();
  }

  /**
   * Returns the shortcut stored in the props file.
   *
   * @param key		the key for the shortcut
   * @return		the shortcut, empty string if not found or none defined
   */
  public String getEditorShortcut(String key) {
    return getPropertiesShortcuts().getProperty("Shortcuts." + key, "");
  }

  /**
   * Returns the shortcut for the tree stored in the props file.
   *
   * @param key		the key for the tree shortcut
   * @return		the shortcut, empty string if not found or none defined
   */
  public String getTreeShortcut(String key) {
    return getPropertiesShortcuts().getProperty("Tree.Shortcuts." + key, "");
  }

  /**
   * Returns whether we can proceed with the operation or not, depending on
   * whether the user saved the flow or discarded the changes.
   *
   * @return		true if safe to proceed
   */
  protected boolean checkForModified() {
    boolean 	result;
    int		retVal;
    String	msg;

    if (!hasCurrentPanel())
      return true;

    result = !getCurrentPanel().isModified();

    if (!result) {
      if (getCurrentPanel().getCurrentFile() == null)
	msg = "Flow not saved - save?";
      else
	msg = "Flow '" + getCurrentPanel().getCurrentFile() + "' not saved - save?";

      retVal = JOptionPane.showConfirmDialog(
	  this,
	  msg,
	  "Flow not saved",
	  JOptionPane.YES_NO_CANCEL_OPTION);

      switch (retVal) {
	case JOptionPane.YES_OPTION:
	  if (getCurrentPanel().getCurrentFile() != null)
	    save();
	  else
	    saveAs();
	  result = !getCurrentPanel().isModified();
	  break;
	case JOptionPane.NO_OPTION:
	  result = true;
	  break;
	case JOptionPane.CANCEL_OPTION:
	  result = false;
	  break;
      }
    }

    return result;
  }

  /**
   * Adds new panel with the specified actor.
   *
   * @param actor	the actor to display in the new panel
   */
  protected void newFlow(AbstractActor actor) {
    FlowPanel	panel;

    panel = m_FlowPanels.newPanel();
    panel.reset(actor);

    updateActions();
    updateWidgets();

    grabFocus();
  }

  /**
   * Sets the current file.
   *
   * @param value	the file
   */
  protected void setCurrentFile(File value) {
    if (hasCurrentPanel())
      getCurrentPanel().setCurrentFile(value);
  }

  /**
   * Returns the current file in use.
   *
   * @return		the current file, can be null
   */
  public File getCurrentFile() {
    if (hasCurrentPanel())
      return getCurrentPanel().getCurrentFile();
    else
      return null;
  }

  /**
   * Attempts to load the file. If non-existent, then a new flow will be
   * created and the current filename set to the provided one.
   *
   * @param file	the file to load
   */
  public void loadUnsafe(File file) {
    FlowPanel	panel;

    panel = m_FlowPanels.newPanel();
    if (!file.exists()) {
      panel.reset(new Flow());
      panel.setCurrentFile(new File(file.getAbsolutePath()));
      updateActions();
    }
    else {
      panel.load(file);
    }
  }

  /**
   * Sets the flow to work on.
   *
   * @param flow	the flow to use
   */
  public void setCurrentFlow(AbstractActor flow) {
    if (hasCurrentPanel())
      getCurrentPanel().setCurrentFlow(flow);
  }

  /**
   * Returns whether a flow panel is available.
   *
   * @return		true if flow panel available
   */
  public boolean hasCurrentPanel() {
    return m_FlowPanels.hasCurrentPanel();
  }

  /**
   * Returns the current flow panel.
   *
   * @return		the current flow panel, null if not available
   */
  public FlowPanel getCurrentPanel() {
    return m_FlowPanels.getCurrentPanel();
  }

  /**
   * Returns the current flow.
   *
   * @return		the current flow, null if not available
   */
  public AbstractActor getCurrentFlow() {
    if (hasCurrentPanel())
      return getCurrentPanel().getTree().getActor();
    else
      return null;
  }

  /**
   * Returns the last flow executed (currently selected flow).
   *
   * @return		the last executed flow, null if not available
   */
  public AbstractActor getLastFlow() {
    if (hasCurrentPanel())
      return getCurrentPanel().getLastFlow();
    else
      return null;
  }

  /**
   * Sets whether the flow is modified or not.
   *
   * @param value	true if the flow is to be flagged as modified
   */
  public void setModified(boolean value) {
    if (hasCurrentPanel())
      getCurrentPanel().getTree().setModified(value);
    update();
  }

  /**
   * Returns whether the flow is flagged as modified.
   *
   * @return		true if the flow is modified
   */
  public boolean isModified() {
    return hasCurrentPanel() && getCurrentPanel().getTree().isModified();
  }

  /**
   * Adds a new tab.
   */
  public void newTab() {
    m_FlowPanels.newPanel();
  }

  /**
   * Opens a flow.
   */
  protected void open() {
    int		retVal;
    FlowPanel	panel;

    retVal = m_FileChooser.showOpenDialog(this);
    if (retVal != BaseFileChooser.APPROVE_OPTION)
      return;

    for (PlaceholderFile file: m_FileChooser.getSelectedPlaceholderFiles()) {
      panel = m_FlowPanels.newPanel();
      panel.load(file);
    }
  }

  /**
   * Reverts a flow.
   */
  protected void revert() {
    if (!hasCurrentPanel())
      return;
    if (!checkForModified())
      return;

    getCurrentPanel().revert();
  }

  /**
   * Saves the flow.
   */
  protected void save() {
    FlowPanel	panel;

    panel = getCurrentPanel();
    if (panel == null)
      return;

    if (panel.getCurrentFile() == null) {
      saveAs();
      return;
    }

    panel.save(panel.getCurrentFile());
  }

  /**
   * Saves the flow.
   */
  protected void saveAs() {
    int		retVal;
    File	file;
    FlowPanel	panel;

    panel = getCurrentPanel();
    if (panel == null)
      return;

    file = panel.getCurrentFile();
    if (file == null)
      file = new PlaceholderFile(getCurrentDirectory() + File.separator + panel.getTitle() + "." + AbstractActor.FILE_EXTENSION);
    if (file.exists())
      file = m_FilenameProposer.propose(file);
    m_FileChooser.setSelectedFile(file);
    retVal = m_FileChooser.showSaveDialog(this);
    if (retVal != BaseFileChooser.APPROVE_OPTION)
      return;

    file = m_FileChooser.getSelectedPlaceholderFile();
    panel.addUndoPoint("Saving undo data...", "Saving as '" + file.getName() + "'");
    showStatus("Saving as '" + file + "'...");

    panel.save(m_FileChooser.getSelectedPlaceholderFile());
  }

  /**
   * Imports a flow.
   */
  protected void importFlow() {
    FlowPanel	panel;

    if (m_ImportDialog == null) {
      if (getParentDialog() != null)
	m_ImportDialog = new ImportDialog(getParentDialog(), ModalityType.DOCUMENT_MODAL);
      else
	m_ImportDialog = new ImportDialog(getParentFrame(), true);
    }

    m_ImportDialog.setLocationRelativeTo(this);
    m_ImportDialog.setVisible(true);
    if (m_ImportDialog.getOption() != ImportDialog.APPROVE_OPTION)
      return;

    panel = m_FlowPanels.newPanel();
    panel.importFlow(m_ImportDialog.getConsumer(), m_ImportDialog.getFile());
  }

  /**
   * Exports the flow.
   */
  protected void exportFlow() {
    FlowPanel	panel;

    panel = getCurrentPanel();
    if (panel == null)
      return;

    if (m_ExportDialog == null) {
      if (getParentDialog() != null)
	m_ExportDialog = new ExportDialog(getParentDialog(), ModalityType.DOCUMENT_MODAL);
      else
	m_ExportDialog = new ExportDialog(getParentFrame(), true);
    }

    m_ExportDialog.setLocationRelativeTo(this);
    m_ExportDialog.setVisible(true);
    if (m_ExportDialog.getOption() != ExportDialog.APPROVE_OPTION)
      return;

    panel.exportFlow(m_ExportDialog.getProducer(), m_ExportDialog.getFile());
  }

  /**
   * Validates the current setup.
   */
  protected void validateSetup() {
    if (hasCurrentPanel())
      getCurrentPanel().validateSetup();
  }

  /**
   * Executes the flow.
   */
  public void run() {
    run(true);
  }

  /**
   * Executes the flow.
   *
   * @param showNotification	whether to show notifications about
   * 				errors/stopped/finished
   */
  public void run(boolean showNotification) {
    if (hasCurrentPanel())
      getCurrentPanel().run(showNotification);
  }

  /**
   * Returns whether a flow is currently running.
   *
   * @return		true if a flow is being executed
   */
  public boolean isRunning() {
    if (getCurrentPanel() == null)
      return false;
    else
      return getCurrentPanel().isRunning();
  }

  /**
   * Returns whether any flow is currently running.
   *
   * @return		true if at least one flow is being executed
   */
  public boolean isAnyRunning() {
    boolean	result;
    int		i;

    result = false;

    for (i = 0; i < m_FlowPanels.getPanelCount(); i++) {
      if (m_FlowPanels.getPanelAt(i).isRunning()) {
	result = true;
	break;
      }
    }

    return result;
  }

  /**
   * Returns whether a flow is currently being stopped.
   *
   * @return		true if a flow is currently being stopped
   */
  public boolean isStopping() {
    if (getCurrentPanel() == null)
      return false;
    else
      return getCurrentPanel().isStopping();
  }

  /**
   * Returns whether any flow is currently stopping.
   *
   * @return		true if at least one flow is being stopped
   */
  public boolean isAnyStopping() {
    boolean	result;
    int		i;

    result = false;

    for (i = 0; i < m_FlowPanels.getPanelCount(); i++) {
      if (m_FlowPanels.getPanelAt(i).isStopping()) {
	result = true;
	break;
      }
    }

    return result;
  }

  /**
   * Returns whether a swing worker is currently running.
   *
   * @return		true if a swing worker is being executed
   */
  public boolean isSwingWorkerRunning() {
    if (getCurrentPanel() == null)
      return false;
    else
      return getCurrentPanel().isSwingWorkerRunning();
  }

  /**
   * Returns whether any swing worker is currently running.
   *
   * @return		true if at least one swing worker is being executed
   */
  public boolean isAnySwingWorkerRunning() {
    boolean	result;
    int		i;

    result = false;

    for (i = 0; i < m_FlowPanels.getPanelCount(); i++) {
      if (m_FlowPanels.getPanelAt(i).isSwingWorkerRunning()) {
	result = true;
	break;
      }
    }

    return result;
  }

  /**
   * Pauses/resumes the flow.
   */
  protected void pauseAndResume() {
    Pausable	pausable;

    if (getCurrentPanel() == null)
      return;

    pausable = (Pausable) getCurrentPanel();
    if (!pausable.isPaused()) {
      showStatus("Pausing");
      m_ActionExecutionPauseAndResume.setName("Resume");
      pausable.pauseExecution();
    }
    else {
      showStatus("Resuming");
      m_ActionExecutionPauseAndResume.setName("Pause");
      pausable.resumeExecution();
    }

    updateActions();
  }

  /**
   * Stops the flow.
   */
  public void stop() {
    getCurrentPanel().stop();
  }

  /**
   * Displays the errors from the last run.
   */
  public void displayErrors() {
    if (hasCurrentPanel())
      getCurrentPanel().displayErrors();
  }

  /**
   * Cleans up the last flow that was run.
   */
  public void cleanUp() {
    m_FlowPanels.cleanUp();
  }

  /**
   * Shows the preferences.
   */
  protected void preferences() {
    PropertiesPreferencesDialog	dialog;

    if (getParentDialog() != null)
      dialog = new PropertiesPreferencesDialog(getParentDialog());
    else
      dialog = new PropertiesPreferencesDialog(getParentFrame());
    dialog.addPreferenceType("InitialDir", PreferenceType.DIRECTORY);
    dialog.addPreferenceType("MaxRecentFlows", PreferenceType.INTEGER);
    dialog.addPreferenceType("ShowQuickInfo", PreferenceType.BOOLEAN);
    dialog.addPreferenceType("ShowAnnotations", PreferenceType.BOOLEAN);
    dialog.addPreferenceType("ShowInputOutput", PreferenceType.BOOLEAN);
    dialog.addPreferenceType("DividerLocation", PreferenceType.INTEGER);
    dialog.addPreferenceType("Tree.ActorName.Size", PreferenceType.INTEGER);
    dialog.addPreferenceType("Tree.ActorName.Color", PreferenceType.COLOR);
    dialog.addPreferenceType("Tree.QuickInfo.Size", PreferenceType.INTEGER);
    dialog.addPreferenceType("Tree.QuickInfo.Color", PreferenceType.COLOR);
    dialog.addPreferenceType("Tree.Annotations.Size", PreferenceType.INTEGER);
    dialog.addPreferenceType("Tree.Annotations.Color", PreferenceType.COLOR);
    dialog.addPreferenceType("Tree.InputOutput.Size", PreferenceType.INTEGER);
    dialog.addPreferenceType("Tree.InputOutput.Color", PreferenceType.COLOR);
    dialog.addPreferenceType("Tree.Placeholders.Size", PreferenceType.INTEGER);
    dialog.addPreferenceType("Tree.Placeholders.Color", PreferenceType.COLOR);
    dialog.addPreferenceType("Tree.VariableHighlight.Background", PreferenceType.COLOR);
    dialog.addPreferenceType("Tree.StateUsesNested", PreferenceType.BOOLEAN);
    dialog.addPreferenceType("Tree.IconScaleFactor", PreferenceType.DOUBLE);
    dialog.addPreferenceType("ClassTree.ShowGlobalInfo", PreferenceType.BOOLEAN);
    dialog.addPreferenceType("StatusBar.Width", PreferenceType.INTEGER);
    dialog.addPreferenceType("StatusBar.Height", PreferenceType.INTEGER);
    dialog.setPreferences(Environment.getInstance().read(FlowEditorPanelDefinition.KEY));
    dialog.setLocationRelativeTo(this);
    dialog.setVisible(true);
    if (dialog.getOption() != PropertiesPreferencesDialog.APPROVE_OPTION)
      return;
    if (!Environment.getInstance().write(FlowEditorPanelDefinition.KEY, dialog.getPreferences())) {
      GUIHelper.showErrorMessage(this, "Failed to save preferences!");
    }
    else {
      m_Properties = null;
      GUIHelper.showInformationMessage(this, "Successfully saved preferences - please re-open the Flow editor!");
    }
  }

  /**
   * Used by the close() method to re-display the flow, in case the flow
   * cannot or should not be closed after all.
   *
   * @see	#close()
   */
  protected void setVisibleAgain() {
    if (getParentDialog() != null)
      getParentDialog().setVisible(true);
    else if (getParentFrame() != null)
      getParentFrame().setVisible(true);
  }

  /**
   * Closes the dialog or frame. But only if no flows are running, being stopped
   * or are modified (in the latter, the user can choose to save the flow).
   */
  protected void close() {
    if (isAnyRunning()) {
      GUIHelper.showErrorMessage(this, "Flows are being executed - closing cancelled!");
      setVisibleAgain();
      return;
    }

    if (isAnyStopping()) {
      GUIHelper.showErrorMessage(this, "Flows are being stopped - closing cancelled!");
      setVisibleAgain();
      return;
    }

    if (!checkForModified()) {
      setVisibleAgain();
      return;
    }

    cleanUp();

    if (getParentFrame() != null)
      ((JFrame) getParentFrame()).setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

    closeParent();
  }

  /**
   * Displays statistics about the current flow.
   */
  protected void statistics() {
    ActorStatistic			stats;
    InformativeStatisticFactory.Dialog	dialog;
    Vector<InformativeStatistic>	statsList;

    if (!hasCurrentPanel())
      return;

    if (getCurrentPanel().getTree().getSelectedNode() != null)
      stats = new ActorStatistic(getCurrentPanel().getTree().getSelectedNode().getFullActor());
    else if (getCurrentFlow() != null)
      stats = new ActorStatistic(getCurrentFlow());
    else
      stats = new ActorStatistic(getCurrentPanel().getTree().getActor());
    statsList = new Vector<InformativeStatistic>();
    statsList.add(stats);

    if (getParentDialog() != null)
      dialog = InformativeStatisticFactory.getDialog(getParentDialog(), ModalityType.DOCUMENT_MODAL);
    else
      dialog = InformativeStatisticFactory.getDialog(getParentFrame(), true);
    dialog.setStatistics(statsList);
    dialog.setTitle("Actor statistics");
    dialog.pack();
    dialog.setLocationRelativeTo(this);
    dialog.setVisible(true);
  }

  /**
   * peforms an undo if possible.
   */
  public void undo() {
    FlowPanel	panel;

    panel = getCurrentPanel();
    if (panel == null)
      return;

    panel.undo();
  }

  /**
   * peforms a redo if possible.
   */
  public void redo() {
    FlowPanel	panel;

    panel = getCurrentPanel();
    if (panel == null)
      return;

    panel.redo();
  }

  /**
   * Searches for actor names in the tree.
   */
  protected void find() {
    if (hasCurrentPanel())
      getCurrentPanel().getTree().find();
  }

  /**
   * Searches for the next actor in the tree.
   */
  protected void findNext() {
    if (hasCurrentPanel())
      getCurrentPanel().getTree().findNext();
  }

  /**
   * Locates an actor based on the full actor name.
   */
  protected void locateActor() {
    String	path;

    path = JOptionPane.showInputDialog("Please enter the full name of the actor (e.g., 'Flow[0].Sequence[3].Display'):");
    if (path == null)
      return;

    if (hasCurrentPanel())
      getCurrentPanel().getTree().locateAndDisplay(path);
  }

  /**
   * Highlights variables in the tree (or hides the highlights again).
   *
   * @param highlight	whether to turn the highlights on or off
   */
  protected void highlightVariables(boolean highlight) {
    String	regexp;

    if (!hasCurrentPanel())
      return;

    if (highlight) {
      regexp = JOptionPane.showInputDialog(
	  this,
	  "Enter the regular expression for the variable name ('.*' matches all):",
	  m_LastVariableSearch);
      if (regexp == null)
	return;

      m_LastVariableSearch = regexp;
      getCurrentPanel().getTree().highlightVariables(m_LastVariableSearch);
    }
    else {
      getCurrentPanel().getTree().highlightVariables(null);
    }
  }

  /**
   * Cleans up the flow, e.g., removing disabled actors, unused global actors.
   *
   * @see		ActorUtils#cleanUpFlow(AbstractActor)
   */
  protected void cleanUpFlow() {
    AbstractActor	cleaned;

    if (!hasCurrentPanel())
      return;

    cleaned = ActorUtils.cleanUpFlow(getCurrentPanel().getTree().getActor());

    if (cleaned != null) {
      getCurrentPanel().addUndoPoint("Saving undo data...", "Cleaning up");
      getCurrentPanel().getTree().buildTree(cleaned);
      getCurrentPanel().getTree().setModified(true);
      update();
    }
  }

  /**
   * Processes the actors with a user-specified actor processor.
   */
  protected void processActors() {
    AbstractActorProcessor		processor;
    ModifyingProcessor			modiyfing;
    GraphicalOutputProducingProcessor	graphical;
    BaseDialog				dialog;

    if (!hasCurrentPanel())
      return;

    if (m_DialogProcessActors == null) {
      if (getParentDialog() != null)
	m_DialogProcessActors = new GenericObjectEditorDialog(getParentDialog());
      else
	m_DialogProcessActors = new GenericObjectEditorDialog(getParentFrame());
      m_DialogProcessActors.setTitle("Process actors");
      m_DialogProcessActors.setModalityType(ModalityType.DOCUMENT_MODAL);
      m_DialogProcessActors.getGOEEditor().setCanChangeClassInDialog(true);
      m_DialogProcessActors.getGOEEditor().setClassType(AbstractActorProcessor.class);
      m_DialogProcessActors.setCurrent(new RemoveDisabledActors());
    }
    m_DialogProcessActors.setLocationRelativeTo(this);
    m_DialogProcessActors.setVisible(true);
    if (m_DialogProcessActors.getResult() != GenericObjectEditorDialog.APPROVE_OPTION)
      return;
    processor = (AbstractActorProcessor) m_DialogProcessActors.getCurrent();
    processor.process(getCurrentPanel().getTree().getActor());
    if (processor instanceof ModifyingProcessor) {
      modiyfing = (ModifyingProcessor) processor;
      if (modiyfing.isModified()) {
	getCurrentPanel().addUndoPoint("Saving undo data...", "Processing actors with " + processor.toString());
	getCurrentPanel().getTree().buildTree(modiyfing.getModifiedActor());
	getCurrentPanel().getTree().setModified(true);
	update();
      }
    }
    if (processor instanceof GraphicalOutputProducingProcessor) {
      graphical = (GraphicalOutputProducingProcessor) processor;
      if (graphical.hasGraphicalOutput()) {
	if (getParentDialog() != null)
	  dialog = new BaseDialog(getParentDialog());
	else
	  dialog = new BaseDialog(getParentFrame());
	dialog.setTitle(processor.getClass().getSimpleName());
	dialog.getContentPane().setLayout(new BorderLayout());
	dialog.getContentPane().add(graphical.getGraphicalOutput(), BorderLayout.CENTER);
	dialog.pack();
	dialog.setLocationRelativeTo(this);
	dialog.setVisible(true);
      }
    }
    showMessage("Actors processed!", false);
  }

  /**
   * Enables/disables all breakpoints in the flow (before execution).
   *
   * @param enable	if true then breakpoints get enabled
   */
  protected void enableBreakpoints(boolean enable) {
    if (hasCurrentPanel())
      getCurrentPanel().getTree().enableBreakpoints(enable);
  }

  /**
   * Displays the variables in the current running flow.
   */
  protected void showVariables() {
    if (hasCurrentPanel())
      getCurrentPanel().showVariables();
  }

  /**
   * Displays a new flow editor window/frame.
   *
   * @return		the new editor panel
   */
  public FlowEditorPanel newWindow() {
    return newWindow(null);
  }

  /**
   * Displays a new flow editor window/frame with the specified actor.
   *
   * @param actor	the actor to display, use null to ignore
   * @return		the new editor panel
   */
  public FlowEditorPanel newWindow(AbstractActor actor) {
    FlowEditorPanel 	result;
    ChildFrame 		oldFrame;
    ChildFrame 		newFrame;
    ChildWindow 	oldWindow;
    ChildWindow 	newWindow;

    result    = null;
    oldFrame = (ChildFrame) GUIHelper.getParent(m_Self, ChildFrame.class);
    if (oldFrame != null) {
      newFrame = oldFrame.getNewWindow();
      newFrame.setVisible(true);
      result  = (FlowEditorPanel) newFrame.getContentPane().getComponent(0);
    }
    else {
      oldWindow = (ChildWindow) GUIHelper.getParent(m_Self, ChildWindow.class);
      if (oldWindow != null) {
	newWindow = oldWindow.getNewWindow();
	newWindow.setVisible(true);
	result  = (FlowEditorPanel) newWindow.getContentPane().getComponent(0);
      }
    }

    // use same directory
    if (result != null) {
      result.setCurrentDirectory(getCurrentDirectory());
      if (actor != null) {
	result.setCurrentFlow(actor);
	result.getCurrentPanel().getTree().setModified(true);
      }
      result.update();
    }

    return result;
  }

  /**
   * Displays the source code (in nested format) of the current flow.
   */
  protected void showSource() {
    if (hasCurrentPanel())
      getCurrentPanel().showSource();
  }

  /**
   * Duplicates the current window/frame, including the current flow.
   *
   * @return		the new window
   */
  public Window duplicateTabInNewWindow() {
    Window		result;
    FlowEditorPanel 	panel;
    ChildFrame 		oldFrame;
    ChildFrame 		newFrame;
    ChildWindow 	oldWindow;
    ChildWindow 	newWindow;

    result   = null;
    panel    = null;
    oldFrame = (ChildFrame) GUIHelper.getParent(m_Self, ChildFrame.class);
    if (oldFrame != null) {
      newFrame = oldFrame.getNewWindow();
      newFrame.setVisible(true);
      panel  = (FlowEditorPanel) newFrame.getContentPane().getComponent(0);
      result = newFrame;
    }
    else {
      oldWindow = (ChildWindow) GUIHelper.getParent(m_Self, ChildWindow.class);
      if (oldWindow != null) {
	newWindow = oldWindow.getNewWindow();
	newWindow.setVisible(true);
	panel  = (FlowEditorPanel) newWindow.getContentPane().getComponent(0);
	result = newWindow;
      }
    }

    // copy information
    if (panel != null) {
      panel.setCurrentDirectory(getCurrentDirectory());
      panel.newTab();
      panel.setCurrentFlow(getCurrentPanel().getCurrentFlow());
      panel.setCurrentFile(getCurrentPanel().getCurrentFile());
      panel.setModified(getCurrentPanel().isModified());
      panel.update();
    }

    return result;
  }

  /**
   * Duplicates the current tab.
   *
   * @return		the new panel
   */
  public FlowPanel duplicateTab() {
    FlowPanel	result;
    FlowPanel 	current;

    result  = null;
    current = getCurrentPanel();

    if (current != null) {
      result = m_FlowPanels.newPanel();
      result.setCurrentFlow(current.getCurrentFlow());
      result.setCurrentFile(current.getCurrentFile());
      result.setModified(current.isModified());
    }

    return result;
  }

  /**
   * Displays the message in the status bar in a separate dialog.
   */
  protected void showMessage() {
    if (m_StatusBar.hasStatus())
      showMessage(m_StatusBar.getStatus(), false);
  }

  /**
   * Displays the given message in a separate dialog.
   *
   * @param msg		the message to display
   * @param isError	whether it is an error message
   */
  protected void showMessage(String msg, boolean isError) {
    String	status;

    status = msg.replaceAll(": ", ":\n");

    if (isError)
      GUIHelper.showErrorMessage(this, status, "Error");
    else
      GUIHelper.showInformationMessage(this, status, "Status");
  }

  /**
   * Displays a message.
   *
   * @param msg		the message to display
   */
  public void showStatus(String msg) {
    m_StatusBar.showStatus(msg);
  }

  /**
   * An undo event occurred.
   *
   * @param e		the event
   */
  public void undoOccurred(UndoEvent e) {
    updateActions();
  }

  /**
   * Sets the current directory in the FileChooser use for opening flows.
   *
   * @param value	the new current directory to use
   */
  public void setCurrentDirectory(File value)  {
    m_FileChooser.setCurrentDirectory(value);
    m_FilenameProposer.setDirectory(value.getAbsolutePath());
  }

  /**
   * Returns the current directory set in the FileChooser used for opening the
   * flows.
   *
   * @return		the current directory
   */
  public File getCurrentDirectory() {
    return m_FileChooser.getCurrentDirectory();
  }

  /**
   * Requests that this Component get the input focus, and that this
   * Component's top-level ancestor become the focused Window. This component
   * must be displayable, visible, and focusable for the request to be
   * granted.
   */
  public void grabFocus() {
    if (hasCurrentPanel())
      getCurrentPanel().grabFocus();
  }

  /**
   * Refreshes the tabs.
   */
  public void refreshTabs() {
    m_Tabs.refresh(getCurrentTree());
  }

  /**
   * Returns the tree.
   *
   * @return		the tree, null if none available
   */
  public Tree getCurrentTree() {
    return m_FlowPanels.getCurrentTree();
  }

  /**
   * Returns the recent files handler in use.
   *
   * @return		the handler
   */
  public RecentFilesHandler getRecentFilesHandler() {
    return m_RecentFilesHandler;
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  public Class[] getSendToClasses() {
    return new Class[]{PlaceholderFile.class};
  }

  /**
   * Checks whether something to send is available.
   *
   * @param cls		the classes to retrieve the item for
   * @return		true if an object is available for sending
   */
  public boolean hasSendToItem(Class[] cls) {
    return    SendToActionUtils.isAvailable(PlaceholderFile.class, cls)
           && hasCurrentPanel()
           && !(!getCurrentPanel().getTree().isModified() && (getCurrentPanel().getCurrentFile() == null));
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the classes to retrieve the item for
   * @return		the item to send
   */
  public Object getSendToItem(Class[] cls) {
    PlaceholderFile	result;
    AbstractActor	actor;

    result = null;

    if (!hasCurrentPanel())
      return result;

    if (SendToActionUtils.isAvailable(PlaceholderFile.class, cls)) {
      if (getCurrentPanel().getTree().isModified()) {
	result = SendToActionUtils.nextTmpFile("floweditor", "flow");
	actor  = getCurrentPanel().getTree().getActor();
	ActorUtils.write(result.getAbsolutePath(), actor);
      }
      else if (getCurrentPanel().getCurrentFile() != null) {
	result = new PlaceholderFile(getCurrentPanel().getCurrentFile());
      }
    }

    return result;
  }

  /**
   * Returns the properties that define the editor.
   *
   * @return		the properties
   */
  public static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(FlowEditorPanelDefinition.KEY);

    return m_Properties;
  }

  /**
   * Returns the properties that define the shortcuts in the editor and tree.
   *
   * @return		the properties
   */
  public static synchronized Properties getPropertiesShortcuts() {
    if (m_PropertiesShortcuts == null)
      m_PropertiesShortcuts = Environment.getInstance().read(FlowEditorPanelShortcutsDefinition.KEY);

    return m_PropertiesShortcuts;
  }
}
