/**
 * GenericObjectEditorPopupMenu.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.goe;

import java.awt.BorderLayout;
import java.awt.Dialog.ModalityType;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyEditor;
import java.util.HashSet;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.option.AbstractOptionProducer;
import adams.core.option.NestedProducer;
import adams.core.option.OptionHandler;
import adams.core.option.OptionUtils;
import adams.gui.core.BaseDialog;
import adams.gui.core.GUIHelper;
import adams.gui.core.TextEditorPanel;

/**
 * Generic GOE popup menu, for copy/paste, etc.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4128 $
 */
public class GenericObjectEditorPopupMenu
  extends JPopupMenu {

  /** for serialization. */
  private static final long serialVersionUID = -5216584001020734521L;

  /** listeners that get notified when the user changes the setup. */
  protected HashSet<ChangeListener> m_ChangeListeners;

  /**
   * Initializes the menu.
   *
   * @param editor	the editor this menu belongs to
   * @param comp	the component to use as parent
   */
  public GenericObjectEditorPopupMenu(final PropertyEditor editor, final JComponent comp) {
    super();

    m_ChangeListeners = new HashSet<ChangeListener>();
    JMenuItem item = null;
    boolean hasNested = (editor.getValue() instanceof OptionHandler);
    final boolean customStringRepresentation = (editor instanceof CustomStringRepresentationHandler);
    final String itemText;
    if (customStringRepresentation)
      itemText = "string representation";
    else
      itemText = "setup";

    // copy nested
    if (hasNested) {
      item = new JMenuItem("Copy nested setup", GUIHelper.getIcon("copy.gif"));
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  GUIHelper.copyToClipboard(AbstractOptionProducer.toString(NestedProducer.class, (OptionHandler) editor.getValue()));
	}
      });
      add(item);
    }

    // copy cmdline
    if (customStringRepresentation)
      item = new JMenuItem("Copy " + itemText, GUIHelper.getEmptyIcon());
    else
      item = new JMenuItem("Copy command-line setup", GUIHelper.getEmptyIcon());
    item.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	if (customStringRepresentation)
	  GUIHelper.copyToClipboard(
	      ((CustomStringRepresentationHandler) editor).toCustomStringRepresentation(editor.getValue()));
	else
	  GUIHelper.copyToClipboard(
	      OptionUtils.getCommandLine(editor.getValue()));
      }
    });
    add(item);

    if (getCanChangeClassInDialog(editor) || customStringRepresentation) {
      // paste
      item = new JMenuItem("Paste " + itemText, GUIHelper.getIcon("paste.gif"));
      item.setEnabled(GUIHelper.canPasteFromClipboard());
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  String str = GUIHelper.pasteSetupFromClipboard();
	  try {
	    if (customStringRepresentation)
	      editor.setValue(
		  ((CustomStringRepresentationHandler) editor).fromCustomStringRepresentation(
		      str));
	    else
	      editor.setValue(
		  OptionUtils.forString(
		      Object.class, str));
	    notifyChangeListeners();
	  }
	  catch (Exception ex) {
	    ex.printStackTrace();
	    GUIHelper.showErrorMessage(
		comp, "Error processing " + itemText + " from clipboard:\n" + ex);
	  }
	}
      });
      add(item);

      // enter setup
      item = new JMenuItem("Enter " + itemText + "...", GUIHelper.getIcon("input.png"));
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  final BaseDialog dlg;
	  if (GUIHelper.getParentDialog(comp) != null)
	    dlg = new BaseDialog(GUIHelper.getParentDialog(comp), ModalityType.DOCUMENT_MODAL);
	  else
	    dlg = new BaseDialog(GUIHelper.getParentFrame(comp), true);
	  dlg.setTitle("Enter " + itemText);
	  dlg.getContentPane().setLayout(new BorderLayout());

	  // text editor
	  final TextEditorPanel textpanel = new TextEditorPanel();
	  dlg.getContentPane().add(textpanel, BorderLayout.CENTER);

	  // buttons
	  JButton buttonOK = new JButton("OK");
	  buttonOK.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      dlg.setVisible(false);
	      try {
		if (customStringRepresentation)
		  editor.setValue(
		      ((CustomStringRepresentationHandler) editor).fromCustomStringRepresentation(
			  textpanel.getContent()));
		else
		  editor.setValue(
		      OptionUtils.forString(
			  Object.class, textpanel.getContent()));
		notifyChangeListeners();
	      }
	      catch (Exception ex) {
		ex.printStackTrace();
		GUIHelper.showErrorMessage(
		    comp, "Error parsing " + itemText + ":\n" + ex);
	      }
	    }
	  });
	  JButton buttonCancel = new JButton("Cancel");
	  buttonCancel.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      dlg.setVisible(false);
	    }
	  });
	  JPanel panel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
	  panel.add(buttonOK);
	  panel.add(buttonCancel);
	  dlg.getContentPane().add(panel, BorderLayout.SOUTH);

	  dlg.pack();
	  dlg.setSize(400, 300);
	  dlg.setLocationRelativeTo(comp);
	  dlg.setVisible(true);
	}
      });
      add(item);
    }
  }

  /**
   * Returns whether the class can be changed in the editor.
   *
   * @param editor	the editor to check
   * @return		true if the class can be changed
   */
  protected boolean getCanChangeClassInDialog(PropertyEditor editor) {
    AbstractGenericObjectEditorHandler	handler;

    handler = AbstractGenericObjectEditorHandler.getHandler(editor);

    return handler.getCanChangeClassInDialog(editor);
  }

  /**
   * Adds the listener to the internal list of listeners that get notified when
   * the user changes the setup.
   *
   * @param l		the listener to add
   */
  public void addChangeListener(ChangeListener l) {
    m_ChangeListeners.add(l);
  }

  /**
   * Removes the listener from the internal list of listeners that get notified
   * when the user changes the setup.
   *
   * @param l		the listener to remove
   */
  public void removeChangeListener(ChangeListener l) {
    m_ChangeListeners.remove(l);
  }

  /**
   * Notifies all change listeners that the user modified the setup.
   */
  protected void notifyChangeListeners() {
    ChangeEvent	e;

    e = new ChangeEvent(this);
    for (ChangeListener l: m_ChangeListeners)
      l.stateChanged(e);
  }
}
