/*
 * ContainerManager.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.container;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import adams.gui.event.DataChangeEvent;
import adams.gui.event.DataChangeListener;
import adams.gui.event.DataChangeEvent.Type;

/**
 * A handler for containers.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 1537 $
 * @param <T> the type of container to use
 */
public class ContainerManager<T extends Container>
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = -2758522365930747099L;

  /** the containers. */
  protected Vector<T> m_List;

  /** the listeners for data changes. */
  protected HashSet<DataChangeListener> m_DataChangeListeners;

  /** whether an update is currently in progress and notifications are
   * suppressed. */
  protected boolean m_Updating;

  /**
   * Initializes the manager.
   */
  public ContainerManager() {
    super();

    m_List                = new Vector<T>();
    m_DataChangeListeners = new HashSet<DataChangeListener>();
    m_Updating            = false;
  }

  /**
   * Initiates the start of a larger update.
   *
   * @see		#isUpdating()
   */
  public void startUpdate() {
    m_Updating = true;
  }

  /**
   * Finishes the update.
   *
   * @see		#isUpdating()
   */
  public void finishUpdate() {
    finishUpdate(true);
  }

  /**
   * Finishes the update.
   *
   * @param notify	whether to notify the listeners about the update
   * @see		#isUpdating()
   */
  protected void finishUpdate(boolean notify) {
    m_Updating = false;

    if (notify)
      notifyDataChangeListeners(new DataChangeEvent(this, Type.BULK_UPDATE));
  }

  /**
   * Returns whether an update is currently in progress.
   *
   * @return		true if an update is currently happening
   */
  public boolean isUpdating() {
    return m_Updating;
  }

  /**
   * Clears the container list.
   */
  public void clear() {
    m_List.clear();

    notifyDataChangeListeners(new DataChangeEvent(this, Type.CLEAR));
  }

  /**
   * Returns a new container containing the given payload.
   *
   * @param o		the payload to encapsulate
   * @return		the new container
   */
  public T newContainer(Comparable o) {
    return (T) new Container(this, o);
  }

  /**
   * Checks whether the container is already in the list. Filling in the
   * preAdd-hook, one can avoid clashes.
   *
   * @param o		the container to look for
   * @return		true if the container is already stored
   */
  public boolean contains(T o) {
    boolean	result;
    int	i;

    result = false;

    for (i = 0; i < m_List.size(); i++) {
      if (m_List.get(i).equals(o)) {
        result = true;
        break;
      }
    }

    return result;
  }

  /**
   * A pre-hook for the add method, before a container gets added to the
   * internal list.
   *
   * @param  c	the container to process
   * @return		the processed container
   */
  protected T preAdd(T c) {
    return c;
  }

  /**
   * Adds the given container to the list.
   *
   * @param c		the container to add
   */
  public void add(T c) {
    c.setManager(this);
    c = preAdd(c);
    m_List.add(c);

    notifyDataChangeListeners(new DataChangeEvent(this, Type.ADDITION, m_List.size() - 1));

    postAdd(c);
  }

  /**
   * A post-hook for the add-method, after the container got added to the internal
   * list and the notifications got sent.
   * <p/>
   * Default implementation does nothing.
   *
   * @param c		the container that got added
   */
  public void postAdd(T c) {
  }

  /**
   * Adds all containers from the given collection.
   *
   * @param c		the collection to add
   */
  public void addAll(Collection<T> c) {
    Iterator<T>	iter;
    int[]	indices;
    int		i;

    startUpdate();

    indices = new int[c.size()];
    iter    = c.iterator();
    i       = 0;
    while (iter.hasNext()) {
      add(iter.next());
      indices[i] = m_List.size() - 1;
      i++;
    }

    finishUpdate(false);

    notifyDataChangeListeners(new DataChangeEvent(this, Type.ADDITION, indices));
  }

  /**
   * Returns the container at the specified location.
   *
   * @param index	the index of the container
   * @return		the container
   */
  public T get(int index) {
    return m_List.get(index);
  }

  /**
   * Returns (a copy of) all currently stored containers. Those containers
   * have no manager.
   *
   * @return		all containers
   */
  public Vector<T> getAll() {
    Vector<T>	result;
    T		cont;
    int		i;

    result = new Vector<T>();

    for (i = 0; i < count(); i++) {
      cont = (T) get(i).copy();
      cont.setManager(null);
      result.add(cont);
    }

    return result;
  }

  /**
   * A pre-hook for the set method, before the container replaces the item
   * currently occupying the position.
   *
   * @param index	the position to place the container
   * @param c		the container to set
   * @return		the processed container
   */
  protected T preSet(int index, T c) {
    return c;
  }

  /**
   * Replaces the container at the given position.
   *
   * @param index	the position to replace
   * @param c		the replacement
   * @return		the old container
   */
  public T set(int index, T c) {
    T		result;
    boolean	localUpdating;

    localUpdating = !m_Updating;
    if (localUpdating)
      m_Updating = true;

    result = m_List.set(index, preSet(index, c));

    if (localUpdating)
      m_Updating = false;

    notifyDataChangeListeners(new DataChangeEvent(this, Type.REPLACEMENT, index, result));

    return result;
  }

  /**
   * Removes the container at the specified position.
   *
   * @param index	the index of the container to remove
   * @return		the container that got removed
   */
  public T remove(int index) {
    T		result;

    result = m_List.remove(index);

    notifyDataChangeListeners(new DataChangeEvent(this, Type.REMOVAL, index, result));

    return result;
  }

  /**
   * Returns the number of containers currently stored.
   *
   * @return		the number of containers
   */
  public int count() {
    return m_List.size();
  }

  /**
   * Determines the index of the container.
   *
   * @param c		the container to look for
   * @return		the index of the container or -1 if not found
   */
  public int indexOf(T c) {
    return m_List.indexOf(c);
  }

  /**
   * Adds the listener to the internal list.
   *
   * @param l		the listener to add
   */
  public void addDataChangeListener(DataChangeListener l) {
    m_DataChangeListeners.add(l);
  }

  /**
   * Removes the listener from the internal list.
   *
   * @param l		the listener to remove
   */
  public void removeDataChangeListener(DataChangeListener l) {
    m_DataChangeListeners.remove(l);
  }

  /**
   * Sends all listeners the specified event. Ignored if an update is
   * currently in progress.
   *
   * @param e		the event to send
   * @see		#isUpdating()
   */
  public void notifyDataChangeListeners(DataChangeEvent e) {
    Iterator<DataChangeListener>	iter;

    if (isUpdating())
      return;

    iter = m_DataChangeListeners.iterator();
    while (iter.hasNext())
      iter.next().dataChanged(e);
  }

  /**
   * Returns a string representation of the handler, i.e., all currently
   * stored containers.
   *
   * @return		 a string representation
   */
  public String toString() {
    StringBuffer	result;
    int		i;

    result = new StringBuffer();

    result.append("[");
    for (i = 0; i < count(); i++)
      result.append(get(i).toString());
    result.append("]");

    return result.toString();
  }
}