/*
 * LogPercentageAxisModel.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core.axis;

/**
 * An axis model for displaying ln(percentage) values.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2574 $
 */
public class LogPercentageAxisModel
  extends AbstractAxisModel {

  /** for serialization. */
  private static final long serialVersionUID = 4789707129254053023L;

  /**
   * Checks whether the data range can be handled by the model.
   *
   * @param min		the minimum value
   * @param max		the maximum value
   * @return		false if data range contains negative values
   */
  public boolean canHandle(double min, double max) {
    return (min > 0.0) && (max > 0.0);
  }

  /**
   * Returns the display name of this model.
   *
   * @return		the display name
   */
  public String getDisplayName() {
    return "Log percent";
  }

  /**
   * Returns the display string of the value for the tooltip, for instance.
   *
   * @param value	the value to turn into string
   * @return		the display string
   */
  public String valueToDisplay(double value) {
    String	result;
    double	raw;

    if (Double.isNaN(value)) {
      result = getActualFormatter().format(value);
    }
    else {
      raw    = (value - m_Minimum) / (m_Maximum - m_Minimum) * 100.0;
      result = getActualFormatter().format(Math.log(raw));
    }

    return result;
  }

  /**
   * Returns the value from the display string.
   *
   * @param display	the string to turn into a value
   * @return		the value
   */
  public double displayToValue(String display) {
    Double	result;

    result = getActualFormatter().parse(display);

    if (!result.isNaN()) {
      result = Math.exp(result);
      result /= 100;
      result *= (m_Maximum - m_Minimum);
      result += m_Minimum;
    }

    return result;
  }

  /**
   * Returns the position on the axis for the given value.
   *
   * @param value	the value to get the position for
   * @return		the corresponding position
   */
  public int valueToPos(double value) {
    int	result;
    int	size;
    double	tmp;

    validate();

    size   = getParent().getLength() - 1;
    tmp    = Math.log(value) - Math.log(m_ActualMinimum);
    tmp    = tmp / (Math.log(m_ActualMaximum) - Math.log(m_ActualMinimum));
    tmp    = tmp * size;
    result = (int) Math.round(tmp);

    return result;
  }

  /**
   * Returns the value for the given position on the axis.
   *
   * @param pos	the position to get the corresponding value for
   * @return		the corresponding value
   */
  public double posToValue(int pos) {
    double	result;
    int	size;
    double	tmp;

    validate();

    size   = getParent().getLength() - 1;
    tmp    = (double) pos / (double) size;
    tmp    = tmp * (Math.log(m_ActualMaximum) - Math.log(m_ActualMinimum));
    tmp    = tmp + Math.log(m_ActualMinimum);
    tmp    = Math.exp(tmp);
    result = tmp;

    return result;
  }
}