/**
 * AdamsTestCase.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.test;

import java.io.Serializable;
import java.lang.reflect.Constructor;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.textui.TestRunner;
import adams.core.ClassLocator;
import adams.core.DebugHelper;
import adams.core.Utils;
import adams.core.management.ProcessUtils;
import adams.core.option.OptionHandler;
import adams.gui.scripting.ScriptingEngine;

/**
 * Ancestor for all test cases.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3484 $
 */
public class AdamsTestCase
  extends TestCase {

  /** property indicating whether multi-process tests should be run. */
  public final static String PROPERTY_MULTIPROCESS_ENABLED = "adams.test.multiprocess.enabled";

  /** property defining the environment class to use (see pom.xml files/surefire plugin). */
  public final static String PROPERTY_ENV_CLASS = "adams.env.class";

  /** whether the information about multi-process tests turned on/off has been output. */
  protected static boolean m_MultiProcessInfoOutput;

  /** the helper class for regression. */
  protected Regression m_Regression;

  /** the test class to use. */
  protected AbstractTestHelper m_TestHelper;

  /** whether to run multiprocess tests (some test classes might offer such tests). */
  protected boolean m_MultiProcessEnabled;

  /**
   * Constructs the test case. Called by subclasses.
   *
   * @param name 	the name of the test
   */
  public AdamsTestCase(String name) {
    super(name);
  }

  /**
   * Tries to load the class based on the test class's name.
   *
   * @return		the class that is being tested or null if none could
   * 			be determined
   */
  protected Class getTestedClass() {
    Class	result;

    result = null;

    if (getClass().getName().endsWith("Test")) {
      try {
	result = Class.forName(getClass().getName().replaceAll("Test$", ""));
      }
      catch (Exception e) {
	result = null;
      }
    }

    return result;
  }

  /**
   * Called by JUnit before each test method.
   *
   * @throws Exception if an error occurs.
   */
  protected void setUp() throws Exception {
    String	clsname;
    Class	cls;

    super.setUp();

    // set up the correct environment
    clsname = System.getProperty(PROPERTY_ENV_CLASS, adams.env.Environment.class.getName());
    adams.env.Environment.setEnvironmentClass(Class.forName(clsname));

    cls = getTestedClass();
    if (cls != null)
      m_Regression = new Regression(cls);

    m_TestHelper          = newTestHelper();
    m_MultiProcessEnabled = Boolean.getBoolean(PROPERTY_MULTIPROCESS_ENABLED);
    if (!m_MultiProcessEnabled && !m_MultiProcessInfoOutput) {
      m_MultiProcessInfoOutput = true;
      System.err.println(
	  "Multi-process tests (if any) not enabled.\n"
        + "You can enable them using the following JVM parameter:\n"
	+ "  -D" + PROPERTY_MULTIPROCESS_ENABLED + "=true\n"
	+ "Note: these tests could extend the running time of the tests considerably!");
    }
  }

  /**
   * Called by JUnit after each test method.
   *
   * @throws Exception	if tear-down fails
   */
  protected void tearDown() throws Exception {
    m_Regression = null;

    ScriptingEngine.stopAllEngines();

    super.tearDown();
  }

  /**
   * Returns the test helper class to use.
   *
   * @return		the helper class instance
   */
  protected AbstractTestHelper newTestHelper() {
    return new TestHelper(this, "");
  }

  /**
   * Performs a serializable test on the given class.
   *
   * @param cls		the class to test
   */
  protected void performSerializableTest(Class cls) {
    Object		obj;
    Constructor		constr;

    if (!ClassLocator.hasInterface(Serializable.class, cls))
      return;

    // default constructor?
    constr = null;
    try {
      constr = cls.getConstructor(new Class[0]);
    }
    catch (NoSuchMethodException e) {
      fail("No default constructor, requires custom test method: " + cls.getName());
    }

    // create instance
    obj = null;
    try {
      obj = constr.newInstance(new Object[0]);
    }
    catch (Exception e) {
      fail("Failed to instantiate object using default constructor: " + cls.getName());
    }

    assertNotNull("Serialization failed", Utils.deepCopy(obj));
  }

  /**
   * For classes (with default constructor) that are serializable, are tested
   * whether they are truly serializable.
   */
  public void testSerializable() {
    if (m_Regression != null)
      performSerializableTest(m_Regression.getRegressionClass());
  }

  /**
   * Performs setting the default options, in case the class implements the
   * OptionHandler interface.
   *
   * @param cls		the class to test
   */
  protected void performDefaultOptionsTest(Class cls) {
    Object		obj;
    OptionHandler	handler;
    Constructor		constr;

    if (!ClassLocator.hasInterface(OptionHandler.class, cls))
      return;

    // default constructor?
    constr = null;
    try {
      constr = cls.getConstructor(new Class[0]);
    }
    catch (NoSuchMethodException e) {
      fail("No default constructor, requires custom test method: " + cls.getName());
    }

    // create instance
    obj = null;
    try {
      obj = constr.newInstance(new Object[0]);
    }
    catch (Exception e) {
      fail("Failed to instantiate object using default constructor: " + cls.getName());
    }

    handler = (OptionHandler) obj;
    handler.getOptionManager().setThrowExceptions(true);
    try {
      handler.getOptionManager().setDefaults();
    }
    catch (Exception e) {
      fail("Setting default options failed: " + cls.getName());
    }
  }

  /**
   * For classes (with default constructor) that implement the OptionHandler
   * interface, are tested whether setting the default options works properly.
   */
  public void testDefaultOptions() {
    if (m_Regression != null)
      performDefaultOptionsTest(m_Regression.getRegressionClass());
  }

  /**
   * Runs the specified suite. Used for running the test from commandline.
   *
   * @param suite	the suite to run
   */
  public static void runTest(Test suite) {
    System.out.println("PID: " + ProcessUtils.getVirtualMachinePID());
    TestRunner.run(suite);
  }
}
