/**
 * GUIHelpProducer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Array;
import java.lang.reflect.Method;

import adams.core.AdditionalInformationHandler;
import adams.core.ClassCrossReference;
import adams.core.Utils;
import adams.core.net.HtmlUtils;

/**
 * Generates the help for the GUI, i.e., HTML output.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4081 $
 */
public class HtmlHelpProducer
  extends AbstractOptionProducer<String,StringBuilder>
  implements FileBasedProducer {

  /** for serialization. */
  private static final long serialVersionUID = 7520567844837662391L;

  /** the buffer for assembling the help. */
  protected StringBuilder m_OutputBuffer;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Generates HTML 3 help output, which is used in the GUI.";
  }

  /**
   * Initializes the output data structure.
   *
   * @return		the created data structure
   */
  protected String initOutput() {
    return "";
  }

  /**
   * Initializes the visitor.
   */
  protected void initialize() {
    super.initialize();

    m_OutputBuffer = new StringBuilder();
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output
   */
  public String getOutput() {
    if (m_Output == null)
      m_Output = m_OutputBuffer.toString();

    return m_Output;
  }

  /**
   * Turns the string into HTML. Line feeds are automatically converted
   * into &lt;br&gt;.
   *
   * @param s		the string to convert to HTML
   * @return		the HTML string
   * @see		HtmlUtils#toHTML(String)
   */
  protected String toHTML(String s) {
    return toHTML(s, false);
  }

  /**
   * Turns the string into HTML. Line feeds are automatically converted
   * into &lt;br&gt;.
   *
   * @param s		the string to convert to HTML
   * @param nbsp	whether to convert leading blanks to non-breaking spaces
   * @return		the HTML string
   * @see		HtmlUtils#toHTML(String)
   */
  protected String toHTML(String s, boolean nbsp) {
    return HtmlUtils.toHTML(s, nbsp);
  }

  /**
   * Breaks up the tool tip and adds it to the StringBuilder.
   *
   * @param option	the current option to obtain the data from
   * @param buffer	the buffer to add the tool tip to
   */
  protected void addToolTip(AbstractOption option, StringBuilder buffer) {
    String	text;

    // obtain and add text
    if (option.getToolTipMethod() != null) {
      try {
	text = (String) option.getToolTipMethod().invoke(option.getOptionHandler(), new Object[]{});
	buffer.append("<p>");
	buffer.append(toHTML(text));
	buffer.append("</p>\n");
      }
      catch (Exception e) {
	// this should never happen!
	e.printStackTrace();
      }
    }
  }

  /**
   * Adds additional information about the argument, e.g., the class.
   *
   * @param option	the current option to obtain the data from
   * @param buffer	the buffer to add the information to
   */
  protected void addArgumentInfo(AbstractArgumentOption option, StringBuilder buffer) {
    String	text;
    Method	method;
    Object[]	vals;

    if (option instanceof EnumOption) {
      text = "";
      try {
	method = option.getBaseClass().getMethod("values", new Class[0]);
	vals   = (Object[]) method.invoke(null, new Object[0]);
	text   = Utils.arrayToString(vals).replaceAll(",", "|");
      }
      catch (Exception e) {
	e.printStackTrace();
	text = "Error retrieving enum values";
      }
    }
    else {
      text = option.getBaseClass().getName();
    }

    buffer.append(" &lt;" + toHTML(text) + "&gt;");
  }

  /**
   * Visits a boolean option.
   *
   * @param option	the boolean option
   * @return		the last internal data structure that was generated
   */
  public StringBuilder processOption(BooleanOption option) {
    StringBuilder	result;

    result = new StringBuilder();

    result.append("<li>\n");
    result.append("<b>" + toHTML(option.getProperty()) + "</b>\n");
    result.append("<br>\n");

    // help
    addToolTip(option, result);

    // command-line
    result.append("<table border=\"1\">\n");
    result.append("<tr>\n");
    result.append("<td>command-line</td>");
    result.append("<td><code>-" + toHTML(option.getCommandline()) + "</code></td>\n");
    result.append("</tr>\n");
    result.append("</table>\n");

    result.append("<br>\n");
    result.append("</li>\n");

    m_OutputBuffer.append(result);

    return result;
  }

  /**
   * Visits a class option.
   *
   * @param option	the class option
   * @return		the last internal data structure that was generated
   */
  public StringBuilder processOption(ClassOption option) {
    return processOption((AbstractArgumentOption) option);
  }

  /**
   * Visits an argument option.
   *
   * @param option	the argument option
   * @return		the last internal data structure that was generated
   */
  public StringBuilder processOption(AbstractArgumentOption option) {
    StringBuilder		result;
    String			text;
    int				n;
    Object			defValue;
    AbstractNumericOption	numeric;

    result = new StringBuilder();

    result.append("<li>\n");
    result.append("<b>" + toHTML(option.getProperty()) + "</b>\n");
    result.append("<br>\n");

    // help
    addToolTip(option, result);

    // command-line
    result.append("<table border=\"1\">\n");
    result.append("<tr>\n");
    result.append("<td>command-line</td>");
    result.append("<td><code>-" + toHTML(option.getCommandline()));
    addArgumentInfo(option, result);
    if (option.isMultiple()) {
      result.append(" [");
      result.append("-" + toHTML(option.getCommandline()));
      result.append(" ...]");
    }
    result.append("</code></td>\n");
    result.append("</tr>\n");

    // add default value
    if (option.getOutputDefaultValue()) {
      text     = null;
      defValue = option.getDefaultValue();

      if (defValue == null) {
	text = "null";
      }
      else if (option.isMultiple()) {
	text = "";
	for (n = 0; n < Array.getLength(defValue); n++) {
	  if (n > 0)
	    text += ", ";
	  text += option.toString(Array.get(defValue, n));
	}
      }
      else {
	text = option.toString(defValue);
      }

      result.append("<tr>\n");
      result.append("<td>default</td>");
      result.append("<td><code>" + toHTML(Utils.backQuoteChars(text)) + "</code></td>\n");
      result.append("</tr>\n");
    }

    if (option instanceof AbstractNumericOption) {
      numeric = (AbstractNumericOption) option;
      if (numeric.hasLowerBound()) {
	result.append("<tr>\n");
	result.append("<td>minimum</td>");
	result.append("<td><code>" + numeric.getLowerBound() + "</code></td>\n");
	result.append("</tr>\n");
      }
      if (numeric.hasUpperBound()) {
	result.append("<tr>\n");
	result.append("<td>maximum</td>");
	result.append("<td><code>" + numeric.getUpperBound() + "</code></td>\n");
	result.append("</tr>\n");
      }
    }

    result.append("</table>\n");
    result.append("<br>\n");
    result.append("</li>\n");

    m_OutputBuffer.append(result);

    return result;
  }

  /**
   * Hook method that gets called just before an option gets produced.
   * <p/>
   * Default implementation does nothing
   *
   * @param manager	the option manager
   * @param index	the index of the option
   */
  protected void preProduce(OptionManager manager, int index) {
    super.preProduce(manager, index);

    m_Output = null;
  }

  /**
   * Hook-method before starting visiting options. Adds header and global
   * info to the output buffer.
   */
  protected void preProduce() {
    Method	method;
    String	globalInfo;
    Class[]	cross;
    int		i;

    m_OutputBuffer = new StringBuilder();
    m_OutputBuffer.append("<html>" + "\n");
    m_OutputBuffer.append("<head>\n");
    m_OutputBuffer.append("<title>" + getInput().getClass().getName() + "<title>\n");
    m_OutputBuffer.append("</head>\n");
    m_OutputBuffer.append("\n");
    m_OutputBuffer.append("<body>\n");
    m_OutputBuffer.append("<h2>Name</h2>\n");
    m_OutputBuffer.append("<p><code>" + getInput().getClass().getName() + "</code></p>\n");
    m_OutputBuffer.append("<br>\n");
    m_OutputBuffer.append("\n");

    try {
      method = getInput().getClass().getMethod("globalInfo", new Class[0]);
      if (method != null) {
	globalInfo = (String) method.invoke(getInput(), new Object[0]);
	m_OutputBuffer.append("<h2>Synopsis</h2>\n");
	m_OutputBuffer.append("<p>" + toHTML(globalInfo, true) + "</p>\n");
	m_OutputBuffer.append("<br>\n");
	m_OutputBuffer.append("\n");
      }
    }
    catch (Exception e) {
      // ignored
    }

    if (getInput() instanceof ClassCrossReference) {
      m_OutputBuffer.append("<h2>See also</h2>\n");
      cross = ((ClassCrossReference) getInput()).getClassCrossReferences();
      m_OutputBuffer.append("<ul>\n");
      for (i = 0; i < cross.length; i++)
	m_OutputBuffer.append("<li>" + cross[i].getName() + "</li>\n");  // TODO hyperlink to class reference?
      m_OutputBuffer.append("</ul>\n");
      m_OutputBuffer.append("\n");
    }

    if (getInput() instanceof AdditionalInformationHandler) {
      m_OutputBuffer.append("<h2>Additional information</h2>\n");
      m_OutputBuffer.append("<p>" + toHTML(((AdditionalInformationHandler) getInput()).getAdditionalInformation()) + "</p>\n");
      m_OutputBuffer.append("<br>\n");
      m_OutputBuffer.append("\n");
    }

    m_OutputBuffer.append("<h2>Options</h2>\n");
    m_OutputBuffer.append("<ul>\n");
  }

  /**
   * Hook-method after visiting options.
   * <p/>
   * Default implementation does nothing.
   */
  protected void postProduce() {
    m_OutputBuffer.append("</ul>\n");
    m_OutputBuffer.append("</body>\n");
    m_OutputBuffer.append("</html>\n");
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output, null in case of an error
   */
  public String toString() {
    return getOutput();
  }

  /**
   * Returns the description of the file format.
   *
   * @return		the description
   */
  public String getFileFormat() {
    return "HTML Help";
  }

  /**
   * Returns the default file extension (without the dot).
   *
   * @return		the default extension
   */
  public String getDefaultFileExtension() {
    return "html";
  }

  /**
   * Returns the file extensions (without the dot).
   *
   * @return		the extensions
   */
  public String[] getFileExtensions() {
    return new String[]{"html", "htm"};
  }
}
