/*
 * AbstractFilter.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.data.filter;

import java.util.Vector;

import adams.core.ClassLister;
import adams.core.CleanUpHandler;
import adams.core.Performance;
import adams.core.ShallowCopySupporter;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.AbstractOptionHandler;
import adams.core.option.OptionUtils;
import adams.data.NotesHandler;
import adams.data.container.DataContainer;
import adams.data.id.DatabaseIDHandler;
import adams.data.id.IDHandler;
import adams.multiprocess.Job;
import adams.multiprocess.JobList;
import adams.multiprocess.JobRunner;

/**
 * Abstract base class for filters.
 *
 * Derived classes only have to override the <code>processData()</code>
 * method. The <code>reset()</code> method can be used to reset an
 * algorithms internal state, e.g., after setting options, which invalidate
 * the previously generated data.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 * @param <T> the data type to pass through the filter
 */
public abstract class AbstractFilter<T extends DataContainer>
  extends AbstractOptionHandler
  implements Comparable, CleanUpHandler, ShallowCopySupporter<AbstractFilter> {

  /** for serialization. */
  private static final long serialVersionUID = 3610605513320220903L;

  /**
   * A job class specific to Filters.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4397 $
   */
  public static class FilterJob<T extends DataContainer>
    extends Job {

    /** for serialization. */
    private static final long serialVersionUID = 5544327082749651329L;

    /** the filter to use. */
    protected AbstractFilter m_Filter;

    /** the data to push through the filter. */
    protected T m_Data;

    /** the filtered data. */
    protected T m_FilteredData;

    /**
     * Initializes the job.
     *
     * @param filter	the filter to use for filtering
     * @param data	the data to pass through the filter
     */
    public FilterJob(AbstractFilter filter, T data) {
      super();

      m_Filter       = filter;
      m_Data         = data;
      m_FilteredData = null;
    }

    /**
     * Returns the filter being used.
     *
     * @return		the filter in use
     */
    public AbstractFilter getFilter() {
      return m_Filter;
    }

    /**
     * The input data.
     *
     * @return		the input data
     */
    public T getData() {
      return m_Data;
    }

    /**
     * The output data, if any.
     *
     * @return		the output data, or null in case of an error
     */
    public T getFilteredData() {
      return m_FilteredData;
    }

    /**
     * Checks whether all pre-conditions have been met.
     *
     * @return		null if everything is OK, otherwise an error message
     */
    protected String preProcessCheck() {
      if (m_Filter == null)
	return "No filter set!";

      if (m_Data == null)
	return "No data set!";

      return null;
    }

    /**
     * Does the actual execution of the job.
     */
    protected void process() {
      m_FilteredData = (T) m_Filter.filter(m_Data);
    }

    /**
     * Checks whether all post-conditions have been met.
     *
     * @return		null if everything is OK, otherwise an error message
     */
    protected String postProcessCheck() {
      if (m_FilteredData == null)
	return "Result of filter is null!";

      return null;
    }

    /**
     * Cleans up data structures, frees up memory.
     * Sets the input data to null.
     */
    public void cleanUp() {
      super.cleanUp();

      m_Filter.destroy();
      m_Filter       = null;
      m_FilteredData = null;
      m_Data         = null;
    }

    /**
     * Returns additional information to be added to the error message.
     *
     * @return		the additional information
     */
    protected String getAdditionalErrorInformation() {
      if (m_Data instanceof NotesHandler)
	return ((NotesHandler) m_Data).getNotes().toString();
      else
	return "";
    }

    /**
     * Returns a string representation of the job.
     *
     * @return		a string representation
     */
    public String toString() {
      String	result;

      result = "data:" + m_Data.getID() + ", ";
      if (m_Data instanceof DatabaseIDHandler)
	result += "db-id: " + ((DatabaseIDHandler) m_Data).getDatabaseID() + ", ";
      result += "filter: " + OptionUtils.getCommandLine(m_Filter);

      return result;
    }
  }

  /**
   * Resets the filter.
   * Derived classes must call this method in set-methods of parameters to
   * assure the invalidation of previously generated data.
   */
  public void reset() {
    super.reset();
  }

  /**
   * Cleans up data structures, frees up memory.
   * Sets the input and generated data to null.
   */
  public void cleanUp() {
    reset();
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Calls cleanUp() and cleans up the options.
   */
  public void destroy() {
    cleanUp();
    super.destroy();
  }

  /**
   * Returns the filtered data.
   *
   * @param data	the data to filter
   * @return		the filtered data
   */
  public T filter(T data) {
    T	result;

    checkData(data);
    result = processData(data);

    if (result instanceof IDHandler)
      result.setID(result.getID() + "'");

    if (result instanceof NotesHandler)
      ((NotesHandler) result).getNotes().addProcessInformation(this);

    return result;
  }

  /**
   * The default implementation only checks whether there is any data set.
   *
   * @param data	the data to filter
   */
  protected void checkData(T data) {
    if (data == null)
      throw new IllegalStateException("No input data provided!");
  }

  /**
   * Performs the actual filtering.
   *
   * @param data	the data to filter
   * @return		the filtered data
   */
  protected abstract T processData(T data);

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o 	the object to be compared.
   * @return  	a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   *
   * @throws ClassCastException 	if the specified object's type prevents it
   *         				from being compared to this object.
   */
  public int compareTo(Object o) {
    if (o == null)
      return 1;

    return OptionUtils.getCommandLine(this).compareTo(OptionUtils.getCommandLine(o));
  }

  /**
   * Returns whether the two objects are the same.
   * <p/>
   * Only compares the commandlines of the two objects.
   *
   * @param o	the object to be compared
   * @return	true if the object is the same as this one
   */
  public boolean equals(Object o) {
    return (compareTo(o) == 0);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @return		the shallow copy
   */
  public AbstractFilter shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractFilter shallowCopy(boolean expand) {
    return (AbstractFilter) OptionUtils.shallowCopy(this, expand);
  }

  /**
   * Returns a list with classnames of filters.
   *
   * @return		the filter classnames
   */
  public static String[] getFilters() {
    return ClassLister.getSingleton().getClassnames(AbstractFilter.class);
  }

  /**
   * Instantiates the filter with the given options.
   *
   * @param classname	the classname of the filter to instantiate
   * @param options	the options for the filter
   * @return		the instantiated filter or null if an error occurred
   */
  public static AbstractFilter forName(String classname, String[] options) {
    AbstractFilter	result;

    try {
      result = (AbstractFilter) OptionUtils.forName(AbstractFilter.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the filter from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			filter to instantiate
   * @return		the instantiated filter
   * 			or null if an error occurred
   */
  public static AbstractFilter forCommandLine(String cmdline) {
    return (AbstractFilter) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }

  /**
   * Passes the data through the given filter and returns it.
   *
   * @param filter	the filter to use for filtering
   * @param data	the data to pass through the filter
   * @return		the filtered data
   */
  public static DataContainer filter(AbstractFilter filter, DataContainer data) {
    DataContainer			result;
    Vector<Vector<DataContainer>>	filtered;
    Vector<DataContainer>		dataList;
    Vector<AbstractFilter>		filterList;

    dataList   = new Vector<DataContainer>();
    dataList.add(data);
    filterList = new Vector<AbstractFilter>();
    filterList.add(filter);
    filtered   = filter(filterList, dataList);
    result     = filtered.get(0).get(0);

    return result;
  }

  /**
   * Passes the data through the given filter and returns it. Makes use of
   * multiple cores, i.e., for each dataset a new thread will be run with a
   * copy of the filter.
   *
   * @param filter	the filter to use for filtering (a new filter with the
   * 			same options will be created and used in each thread)
   * @param data	the data to pass through the filter
   * @return		the filtered data, the index corresponds to the
   * 			data index
   */
  public static Vector<DataContainer> filter(AbstractFilter filter, Vector<DataContainer> data) {
    Vector<DataContainer>		result;
    Vector<Vector<DataContainer>>	filtered;
    Vector<DataContainer>		dataList;
    Vector<AbstractFilter>		filterList;

    dataList   = new Vector<DataContainer>();
    dataList.addAll(data);
    filterList = new Vector<AbstractFilter>();
    filterList.add(filter);
    filtered   = filter(filterList, dataList);
    result     = filtered.get(0);

    return result;
  }

  /**
   * Passes the data through the given filters and returns it. Makes use of
   * multiple cores, i.e., for each dataset a new thread will be run with a
   * copy of the filter.
   *
   * @param filter	the filters to use for filtering (a new filter with the
   * 			same options will be created and used in each thread)
   * @param data	the data to pass through the filter
   * @return		the filtered data, the index corresponds to the filter
   * 			index
   */
  public static Vector<DataContainer> filter(Vector<AbstractFilter> filter, DataContainer data) {
    Vector<DataContainer>		result;
    Vector<Vector<DataContainer>>	filtered;
    Vector<DataContainer>		dataList;
    Vector<AbstractFilter>		filterList;
    int					i;

    dataList   = new Vector<DataContainer>();
    dataList.add(data);
    filterList = new Vector<AbstractFilter>();
    filterList.addAll(filter);
    filtered   = filter(filterList, dataList);
    result     = new Vector<DataContainer>();
    for (i = 0; i < filtered.size(); i++)
      result.add(filtered.get(i).get(0));

    return result;
  }

  /**
   * Passes the data through the given filters and returns it. Makes use of
   * multiple cores, i.e., for each dataset a new thread will be run with a
   * copy of the filter.
   *
   * @param filter	the filters to use for filtering (a new filter with the
   * 			same options will be created and used in each thread)
   * @param data	the data to pass through the filter
   * @return		the filtered data, the indices in the outer Vector
   * 			correspond to the filter index, the inner Vector to
   * 			the index of the data
   */
  public static Vector<Vector<DataContainer>> filter(Vector<AbstractFilter> filter, Vector<DataContainer> data) {
    Vector<Vector<DataContainer>>	result;
    Vector<DataContainer>		subresult;
    AbstractFilter			threadFilter;
    JobRunner<FilterJob> 		runner;
    JobList<FilterJob>			jobs;
    FilterJob				job;
    int					i;
    int					n;

    result = new Vector<Vector<DataContainer>>();

    if (Performance.getMultiProcessingEnabled()) {
      runner = new JobRunner<FilterJob>();
      jobs   = new JobList<FilterJob>();

      // fill job list
      for (n = 0; n < filter.size(); n++) {
	for (i = 0; i < data.size(); i++) {
	  threadFilter = filter.get(n).shallowCopy(true);
	  jobs.add(new FilterJob(threadFilter, data.get(i)));
	}
      }
      runner.add(jobs);
      runner.start();
      runner.stop();

      // gather results
      subresult = null;
      for (i = 0; i < jobs.size(); i++) {
	if (i % data.size() == 0) {
	  subresult = new Vector<DataContainer>();
	  result.add(subresult);
	}
	job = jobs.get(i);
	// success? If not, just add the header of the original data
	if (job.getFilteredData() != null)
	  subresult.add(job.getFilteredData());
	else
	  subresult.add(job.getData().getHeader());
	job.cleanUp();
      }
    }
    else {
      for (n = 0; n < filter.size(); n++) {
	subresult = new Vector<DataContainer>();
	result.add(subresult);
	for (i = 0; i < data.size(); i++) {
	  threadFilter = filter.get(n).shallowCopy(true);
	  subresult.add(threadFilter.filter(data.get(i)));
	}
      }
    }

    return result;
  }
}
