/*
 * XYSequence.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.data.sequence;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Iterator;

import adams.core.io.PlaceholderFile;
import adams.data.container.AbstractDataContainer;
import adams.data.container.DataPointComparator;
import adams.data.sequence.XYSequencePointComparator.Comparison;

/**
 * A sequence storing 2-dimensional points.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3921 $
 * @param <X> the type of X
 * @param <Y> the type of Y
 */
public class XYSequence<X extends Number & Comparable, Y extends Number & Comparable>
  extends AbstractDataContainer<XYSequencePoint<X, Y>> {

  /** for serialization. */
  private static final long serialVersionUID = -3742507986126404535L;

  /** the file extension. */
  public final static String FILE_EXTENSION = ".csv";

  /** the default comparator. */
  protected DataPointComparator m_Comparator;

  /** the comparison to use. */
  protected Comparison m_Comparison;

  /**
   * Initializes the sequence.
   */
  public XYSequence() {
    super();

    setComparison(Comparison.X);
  }

  /**
   * Sets the type of comparison to use.
   *
   * @param value	the type of comparison to use
   */
  public void setComparison(Comparison value) {
    m_Comparison = value;
    m_Comparator = newComparator();
  }

  /**
   * Returns the type of comparison currently in use.
   *
   * @return		the type of comparison
   */
  public Comparison getComparison() {
    return m_Comparison;
  }

  /**
   * Returns the comparator in use.
   *
   * @return		the comparator to use
   */
  public DataPointComparator<XYSequencePoint<X, Y>> newComparator() {
    return new XYSequencePointComparator<X, Y>(m_Comparison, true);
  }

  /**
   * Returns the comparator in use.
   *
   * @return		the comparator in use
   */
  public DataPointComparator<XYSequencePoint<X, Y>> getComparator() {
    return m_Comparator;
  }

  /**
   * Returns a new instance of a sequence point.
   *
   * @return		the new sequence point
   */
  public XYSequencePoint<X, Y> newPoint() {
    return new XYSequencePoint<X, Y>();
  }

  /**
   * Returns the SequencePoint with the exact number, null if not found.
   *
   * @param x		the number to look for
   * @return		the SequencePoint or null if not found
   * @see		#findClosest(Number)
   */
  public XYSequencePoint<X, Y> find(X x) {
    XYSequencePoint<X, Y>	result;
    int				index;
    ArrayList<XYSequencePoint>	points;

    result = null;

    points = new ArrayList<XYSequencePoint>(m_Points);  // TODO: inefficient
    index  = XYSequenceUtils.findX(points, (Number) x);
    if (index > -1)
      result = m_Points.get(index);

    return result;
  }

  /**
   * Returns the SequencePoint with a number closest to the one provided.
   *
   * @param x		the number to look for
   * @return		the SpectrumPoint
   * @see		#find(Number)
   */
  public XYSequencePoint<X, Y> findClosest(X x) {
    XYSequencePoint<X, Y>	result;
    int				index;
    ArrayList<XYSequencePoint>	points;

    result = null;

    points = new ArrayList<XYSequencePoint>(m_Points);  // TODO: inefficient
    index  = XYSequenceUtils.findClosestX(points, (Number) x);
    if (index > -1)
      result = m_Points.get(index);

    return result;
  }

  /**
   * Returns the file header.
   *
   * @return		the header
   */
  public String getFileHeader() {
    return "ID,X,Y";
  }

  /**
   * Writes its content to the given file.
   *
   * @param filename	the file to write to
   * @return		true if successfully written
   */
  public boolean write(String filename) {
    boolean		result;
    BufferedWriter	writer;

    try {
      writer = new BufferedWriter(new FileWriter(new PlaceholderFile(filename).getAbsolutePath()));
      result = write(writer);
      writer.close();
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Writes its content with the given writer.
   *
   * @param writer	the writer to use
   * @return		true if successfully written
   */
  public boolean write(BufferedWriter writer) {
    boolean				result;
    Iterator<XYSequencePoint<X,Y>>	iter;

    try {
      // header
      writer.write(getFileHeader());
      writer.newLine();

      // data points
      iter = iterator();
      while (iter.hasNext()) {
	writer.write(iter.next().toString());
	writer.newLine();
      }
      writer.flush();

      result = true;
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Reads its content from the given file.
   *
   * @param filename	the file to read from
   * @return		true if successfully read
   */
  public boolean read(String filename) {
    boolean		result;
    BufferedReader	reader;

    try {
      reader = new BufferedReader(new FileReader(filename));
      read(reader);
      reader.close();
      result = true;
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Reads its content from the given reader.
   *
   * @param reader	the reader to use
   * @return		true if successfully read
   */
  public boolean read(BufferedReader reader) {
    boolean			result;
    XYSequencePoint<X,Y>	template;
    XYSequencePoint<X,Y>	point;
    String			line;

    clear();

    try {
      result   = true;
      template = newPoint();

      // header - ignored
      reader.readLine();

      // data points
      while (((line = reader.readLine()) != null) && result) {
	point  = template.parse(line);
	result = (point != null);
	if (result)
	  add(point);
      }
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Only for testing.
   *
   * @param args	ignored
   * @throws Exception	if something goes wrong
   */
  public static void main(String[] args) throws Exception {
    XYSequence s = new XYSequence();
    s.add(new XYSequencePoint("1", 1.0, 0.1));
    s.add(new XYSequencePoint("2", 2.0, 0.2));
    s.add(new XYSequencePoint("3", 3.0, 0.9));
    s.add(new XYSequencePoint("4", 4.0, 0.5));
    s.add(new XYSequencePoint("5", 5.0, 0.1));

    StringWriter writer = new StringWriter();
    s.write(new BufferedWriter(writer));
    System.out.println(writer.toString());

    StringReader reader = new StringReader(writer.toString());
    XYSequence s2 = new XYSequence();
    s2.read(new BufferedReader(reader));

    System.out.println("s == s2? " + s.equals(s2));
  }
}
