/*
 * XYSequencePoint.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.data.sequence;

import adams.core.Utils;
import adams.data.container.AbstractDataPoint;
import adams.data.container.DataPoint;

/**
 * A 2-dimensional point. With an optional ID string attached to it.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3922 $
 * @param <X> the type of X
 * @param <Y> the type of Y
 */
public class XYSequencePoint<X extends Number & Comparable, Y extends Number & Comparable>
  extends AbstractDataPoint {

  /** for serialization. */
  private static final long serialVersionUID = 2354312871454097142L;

  /** an optional ID. */
  protected String m_ID;

  /** the X value. */
  protected X m_X;

  /** the Y value. */
  protected Y m_Y;

  /**
   * Initializes the point with no points and no ID.
   */
  public XYSequencePoint() {
    this(null, null);
  }

  /**
   * Initializes the point with no ID.
   *
   * @param x		the X value
   * @param y		the Y value
   */
  public XYSequencePoint(X x, Y y) {
    this(null, x, y);
  }

  /**
   * Initializes the point.
   *
   * @param id		the ID, use null to ignore
   * @param x		the X value
   * @param y		the Y value
   */
  public XYSequencePoint(String id, X x, Y y) {
    super();

    setID(id);
    setX(x);
    setY(y);
  }

  /**
   * Sets the X value.
   *
   * @param value	the new X value
   */
  public void setX(X value) {
    m_X = value;
  }

  /**
   * Returns the X value.
   *
   * @return		the X value
   */
  public X getX() {
    return m_X;
  }

  /**
   * Sets the Y value.
   *
   * @param value	the new Y value
   */
  public void setY(Y value) {
    m_Y = value;
  }

  /**
   * Returns the Y value.
   *
   * @return		the Y value
   */
  public Y getY() {
    return m_Y;
  }

  /**
   * Sets the ID.
   *
   * @param value	the new ID
   */
  public void setID(String value) {
    if (value == null)
      m_ID = "";
    else
      m_ID = value;
  }

  /**
   * Returns the ID.
   *
   * @return		the ID
   */
  public String getID() {
    return m_ID;
  }

  /**
   * Checks whether an ID is set.
   *
   * @return		true if an ID is available
   */
  public boolean hasID() {
    return (m_ID.length() > 0);
  }

  /**
   * Compares this object with the specified object for order.  Returns a
   * negative integer, zero, or a positive integer as this object is less
   * than, equal to, or greater than the specified object.
   *
   * @param   o the object to be compared.
   * @return  a negative integer, zero, or a positive integer as this object
   *		is less than, equal to, or greater than the specified object.
   * @throws ClassCastException if the specified object's type prevents it
   *         from being compared to this object.
   */
  public int compareTo(Object o) {
    int			result;
    XYSequencePoint	other;

    if (o == null)
      return 1;
    else
      result = 0;

    if (!(o instanceof XYSequencePoint))
      return -1;

    other = (XYSequencePoint) o;

    result = getID().compareTo(other.getID());

    if (result == 0)
      result = getX().compareTo(other.getX());

    if (result == 0)
      result = getY().compareTo(other.getY());

    return result;
  }

  /**
   * Obtains the stored variables from the other data point.
   *
   * @param other	the data point to get the values from
   */
  public void assign(DataPoint other) {
    XYSequencePoint<X, Y>	point;

    super.assign(other);

    point = (XYSequencePoint<X, Y>) other;

    setX(point.getX());
    setY(point.getY());
  }

  /**
   * Parses a string and instantiates a sequence point of it.
   *
   * @param s		the string to parse
   * @return		the instantiated point, null in case of an error
   */
  public XYSequencePoint parse(String s) {
    XYSequencePoint	result;
    String[]		parts;

    result = null;

    parts = s.split(",");
    if (parts.length == 3)
      result = new XYSequencePoint(
	  		parts[0],
	  		Double.parseDouble(parts[1]),
	  		Double.parseDouble(parts[2]));

    return result;
  }

  /**
   * Returns a string representation of the point.
   *
   * @return		the string representation
   */
  public String toString() {
    return Utils.quote(getID()) + "," + getX() + "," + getY();
  }
}
