/**
 * ProxyPanel.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.dialog;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.net.Proxy;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JSpinner;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.Utils;
import adams.core.net.ProxyHelper;
import adams.gui.core.BasePanel;
import adams.gui.core.ParameterPanel;

/**
 * Panel for configuring the proxy settings.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 1741 $
 */
public class ProxySettingsPanel
  extends BasePanel {

  /** for serialization. */
  private static final long serialVersionUID = -8502296969739181339L;

  /** the panel for the type. */
  protected BasePanel m_PanelType;

  /** the combobox for the proxy type. */
  protected JComboBox m_ComboBoxType;

  /** the panel for the http parameters. */
  protected ParameterPanel m_PanelHttpFtp;

  /** the http proxy host. */
  protected JTextField m_TextHttpFtpHost;

  /** the http proxy port. */
  protected JSpinner m_SpinnerHttpFtpPort;

  /** whether authentication is necessary. */
  protected JCheckBox m_CheckBoxHttpFtpAuthentication;

  /** the http proxy user. */
  protected JTextField m_TextHttpFtpUser;

  /** the http proxy password. */
  protected JTextField m_TextHttpFtpPassword;

  /** the hosts that bypass the http proxy . */
  protected JTextField m_TextHttpFtpNoProxy;

  /** the panel for the socks parameters. */
  protected ParameterPanel m_PanelSocks;

  /** the socks proxy host. */
  protected JTextField m_TextSocksHost;

  /** the socks proxy port. */
  protected JSpinner m_SpinnerSocksPort;

  /** whether authentication is necessary. */
  protected JCheckBox m_CheckBoxSocksAuthentication;

  /** the socks proxy user. */
  protected JTextField m_TextSocksUser;

  /** the socks proxy password. */
  protected JTextField m_TextSocksPassword;

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    JLabel	label;

    super.initGUI();

    setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    setLayout(new BorderLayout());

    // type
    m_PanelType = new BasePanel(new GridLayout(1, 2, 5, 5));
    add(m_PanelType, BorderLayout.NORTH);

    m_ComboBoxType = new JComboBox(Proxy.Type.values());
    m_ComboBoxType.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	// http/ftp
	m_PanelHttpFtp.setEnabled(m_ComboBoxType.getSelectedItem().equals(Proxy.Type.HTTP));
	boolean sel = m_CheckBoxHttpFtpAuthentication.isSelected();
	m_CheckBoxHttpFtpAuthentication.setSelected(!sel);
	m_CheckBoxHttpFtpAuthentication.setSelected(sel);
	// socks
	m_PanelSocks.setEnabled(m_ComboBoxType.getSelectedItem().equals(Proxy.Type.SOCKS));
	sel = m_CheckBoxSocksAuthentication.isSelected();
	m_CheckBoxSocksAuthentication.setSelected(!sel);
	m_CheckBoxSocksAuthentication.setSelected(sel);
      }
    });
    label = new JLabel("Connection");
    label.setLabelFor(m_ComboBoxType);
    m_PanelType.add(label);
    m_PanelType.add(m_ComboBoxType);

    // http+ftp
    m_PanelHttpFtp = new ParameterPanel();
    m_PanelHttpFtp.setBorder(BorderFactory.createTitledBorder("Http & Ftp"));
    add(m_PanelHttpFtp, BorderLayout.CENTER);

    m_TextHttpFtpHost = new JTextField(15);
    m_PanelHttpFtp.addParameter("_Host", m_TextHttpFtpHost);

    m_SpinnerHttpFtpPort = new JSpinner();
    m_PanelHttpFtp.addParameter("Port", m_SpinnerHttpFtpPort);

    m_TextHttpFtpNoProxy = new JTextField(15);
    m_PanelHttpFtp.addParameter("No pro_xy for", m_TextHttpFtpNoProxy);

    m_CheckBoxHttpFtpAuthentication = new JCheckBox();
    m_CheckBoxHttpFtpAuthentication.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	m_TextHttpFtpUser.setEnabled(m_CheckBoxHttpFtpAuthentication.isSelected());
	m_TextHttpFtpPassword.setEnabled(m_CheckBoxHttpFtpAuthentication.isSelected());
      }
    });
    m_PanelHttpFtp.addParameter("Requires _authentication", m_CheckBoxHttpFtpAuthentication);

    m_TextHttpFtpUser = new JTextField(15);
    m_TextHttpFtpUser.setEnabled(false);
    m_PanelHttpFtp.addParameter("User", m_TextHttpFtpUser);

    m_TextHttpFtpPassword = new JTextField(15);
    m_TextHttpFtpPassword.setEnabled(false);
    m_PanelHttpFtp.addParameter("Password", m_TextHttpFtpPassword);

    // socks
    m_PanelSocks = new ParameterPanel();
    m_PanelSocks.setBorder(BorderFactory.createTitledBorder("Socks"));
    add(m_PanelSocks, BorderLayout.SOUTH);

    m_TextSocksHost = new JTextField(15);
    m_PanelSocks.addParameter("_Host", m_TextSocksHost);

    m_SpinnerSocksPort = new JSpinner();
    m_PanelSocks.addParameter("Port", m_SpinnerSocksPort);

    m_CheckBoxSocksAuthentication = new JCheckBox();
    m_CheckBoxSocksAuthentication.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	m_TextSocksUser.setEnabled(m_CheckBoxSocksAuthentication.isSelected());
	m_TextSocksPassword.setEnabled(m_CheckBoxSocksAuthentication.isSelected());
      }
    });
    m_PanelSocks.addParameter("Requires _authentication", m_CheckBoxSocksAuthentication);

    m_TextSocksUser = new JTextField(15);
    m_TextSocksUser.setEnabled(false);
    m_PanelSocks.addParameter("User", m_TextSocksUser);

    m_TextSocksPassword = new JTextField(15);
    m_TextSocksPassword.setEnabled(false);
    m_PanelSocks.addParameter("Password", m_TextSocksPassword);

    // display values
    load();
  }

  /**
   * Loads the values from the props file and displays them.
   */
  protected void load() {
    ProxyHelper	proxy;

    proxy = ProxyHelper.getSingleton();
    proxy.reload();

    // type
    m_ComboBoxType.setSelectedItem(proxy.getProxyType());

    // http
    m_TextHttpFtpHost.setText(proxy.getHost(Proxy.Type.HTTP));
    m_SpinnerHttpFtpPort.setValue(proxy.getPort(Proxy.Type.HTTP));
    m_TextHttpFtpNoProxy.setText(Utils.flatten(proxy.getNoProxy(Proxy.Type.HTTP), ", "));
    m_CheckBoxHttpFtpAuthentication.setSelected(proxy.getAuthentication(Proxy.Type.HTTP));
    m_TextHttpFtpUser.setText(proxy.getUser(Proxy.Type.HTTP));
    m_TextHttpFtpPassword.setText(proxy.getPassword(Proxy.Type.HTTP));

    // socks
    m_TextSocksHost.setText(proxy.getHost(Proxy.Type.SOCKS));
    m_SpinnerSocksPort.setValue(proxy.getPort(Proxy.Type.SOCKS));
    m_CheckBoxSocksAuthentication.setSelected(proxy.getAuthentication(Proxy.Type.SOCKS));
    m_TextSocksUser.setText(proxy.getUser(Proxy.Type.SOCKS));
    m_TextSocksPassword.setText(proxy.getPassword(Proxy.Type.SOCKS));
  }

  /**
   * Activates the proxy settings.
   */
  public void activate() {
    ProxyHelper	proxy;

    proxy = ProxyHelper.getSingleton();

    // type
    if (m_ComboBoxType.getSelectedIndex() > -1)
      proxy.setProxyType(Proxy.Type.values()[m_ComboBoxType.getSelectedIndex()]);
    else
      proxy.setProxyType(Proxy.Type.DIRECT);

    // http
    proxy.setHost(Proxy.Type.HTTP, m_TextHttpFtpHost.getText());
    proxy.setPort(Proxy.Type.HTTP, ((Number) m_SpinnerHttpFtpPort.getValue()).intValue());
    proxy.setNoProxy(Proxy.Type.HTTP, m_TextHttpFtpNoProxy.getText().replaceAll(" ", "").split(","));
    proxy.setAuthentication(Proxy.Type.HTTP, m_CheckBoxHttpFtpAuthentication.isSelected());
    proxy.setUser(Proxy.Type.HTTP, m_TextHttpFtpUser.getText());
    proxy.setPassword(Proxy.Type.HTTP, m_TextHttpFtpPassword.getText());

    // socks
    proxy.setHost(Proxy.Type.SOCKS, m_TextSocksHost.getText());
    proxy.setPort(Proxy.Type.SOCKS, ((Number) m_SpinnerSocksPort.getValue()).intValue());
    proxy.setAuthentication(Proxy.Type.SOCKS, m_CheckBoxSocksAuthentication.isSelected());
    proxy.setUser(Proxy.Type.SOCKS, m_TextSocksUser.getText());
    proxy.setPassword(Proxy.Type.SOCKS, m_TextSocksPassword.getText());

    proxy.save();
    proxy.initializeProxy();
  }
}
