/**
 * ScriptingEngine.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.scripting;

import java.util.Iterator;

import adams.core.DebugHelper;
import adams.core.Properties;
import adams.db.AbstractDatabaseConnection;
import adams.db.DatabaseConnection;
import adams.env.ScriptingEngineDefinition;

/**
 * Processes scripting commands.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 */
public class ScriptingEngine
  extends AbstractScriptingEngine {

  /** for serialization. */
  private static final long serialVersionUID = -2966869686762723507L;

  /** the scripting engine manager. */
  private static ScriptingEngineManager m_ScriptingEngineManager;

  /** the properties for scripting. */
  private static Properties m_Properties;

  /**
   * Returns the debugging level (0 = turned off).
   *
   * @return		the debugging level
   */
  protected int initDebugLevel() {
    return DebugHelper.getDebugLevel(ScriptingEngine.class);
  }

  /**
   * Returns the default database connection.
   *
   * @return		the default database connection
   */
  protected AbstractDatabaseConnection getDefaultDatabaseConnection() {
    return DatabaseConnection.getSingleton();
  }

  /**
   * Returns the properties key to use for retrieving the properties.
   *
   * @return		the key
   */
  protected String getDefinitionKey() {
    return ScriptingEngineDefinition.KEY;
  }

  /**
   * Provides access to the properties object.
   *
   * @return		the properties
   */
  protected synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = readProperties();

    return m_Properties;
  }

  /**
   * Returns the singleton instance of the scripting engine.
   *
   * @param dbcon	the database context
   * @return		the singleton
   */
  public synchronized static AbstractScriptingEngine getSingleton(AbstractDatabaseConnection dbcon) {
    if (m_ScriptingEngineManager == null)
      m_ScriptingEngineManager = new ScriptingEngineManager();
    if (!m_ScriptingEngineManager.has(dbcon)) {
      m_ScriptingEngineManager.add(dbcon, new ScriptingEngine());
      m_ScriptingEngineManager.get(dbcon).setDatabaseConnection(dbcon);
    }

    return m_ScriptingEngineManager.get(dbcon);
  }

  /**
   * Stops all scripting engines.
   */
  public synchronized static void stopAllEngines() {
    Iterator<AbstractScriptingEngine>	iter;
    if (m_ScriptingEngineManager != null) {
      iter = m_ScriptingEngineManager.iterator();
      while (iter.hasNext())
	iter.next().stopEngine();
    }
  }
}
