/**
 * XYSequencePaintletWithFixedXRange.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.sequence;

import java.awt.Graphics;

import adams.gui.visualization.core.PaintablePanel;

/**
 * A wrapper for XY-sequence paintlets, in order to use fixed a X range.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3536 $
 */
public class XYSequencePaintletWithFixedXRange
  extends AbstractXYSequencePaintlet
  implements PaintletWithFixedXRange {

  /** for serialization. */
  private static final long serialVersionUID = 3270329510617886683L;

  /** the minimum of X. */
  protected double m_MinX;

  /** the maximum of X. */
  protected double m_MaxX;

  /** the actual paintlet to use. */
  protected AbstractXYSequencePaintlet m_Paintlet;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Meta-paintlet that uses a fixed X range (for faster drawing) and a base-paintlet to draw the actual data.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "min-x", "minX",
	    0.0, null, null);

    m_OptionManager.add(
	    "max-x", "maxX",
	    1000.0, null, null);

    m_OptionManager.add(
	    "paintlet", "paintlet",
	    getDefaultPaintlet());
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    setPaintlet(getDefaultPaintlet());

    super.initialize();
  }

  /**
   * Returns the default paintlet to use.
   *
   * @return		the default paintlet
   */
  protected AbstractXYSequencePaintlet getDefaultPaintlet() {
    return new XYSequenceLinePaintlet();
  }

  /**
   * Sets the spectrum panel to use, null to disable painting.
   *
   * @param value	the panel to paint on
   */
  public void setPanel(PaintablePanel value) {
    if (m_Paintlet != null)
      m_Paintlet.setPanel(value);

    super.setPanel(value);
  }

  /**
   * Sets the minimum of the X range.
   *
   * @param value	the minimum
   */
  public void setMinX(double value) {
    m_MinX = value;
    memberChanged(true);
  }

  /**
   * Returns the minimum of the X range.
   *
   * @return		the minimum
   */
  public double getMinX() {
    return m_MinX;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String minXTipText() {
    return "The minimum value for the X range.";
  }

  /**
   * Sets the maximum of the X range.
   *
   * @param value	the maximum
   */
  public void setMaxX(double value) {
    m_MaxX = value;
    memberChanged(true);
  }

  /**
   * Returns the maximum of the X range.
   *
   * @return		the maximum
   */
  public double getMaxX() {
    return m_MaxX;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String maxXTipText() {
    return "The maximum value for the X range.";
  }

  /**
   * Sets the actual paintlet to use.
   *
   * @param value	the paintlet
   */
  public void setPaintlet(AbstractXYSequencePaintlet value) {
    if (m_Paintlet != null)
      m_Paintlet.setPanel(null);

    m_Paintlet = value;
    m_Paintlet.setPanel(getPanel());

    memberChanged();
  }

  /**
   * Returns the painlet in use.
   *
   * @return		the paintlet
   */
  public AbstractXYSequencePaintlet getPaintlet() {
    return m_Paintlet;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String paintletTipText() {
    return "The actual paintlet to use for drawing the data.";
  }

  /**
   * The paint routine of the paintlet.
   *
   * @param g		the graphics context to use for painting
   */
  public void performPaint(Graphics g) {
    m_Paintlet.performPaint(g);
  }

  /**
   * Returns a new instance of the hit detector to use.
   *
   * @return		the hit detector
   */
  public AbstractXYSequencePointHitDetector newHitDetector() {
    return m_Paintlet.newHitDetector();
  }
}
