/*
 * FileUtilsTest.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.core.io;

import java.io.File;

import junit.framework.Test;
import junit.framework.TestSuite;
import adams.core.io.FileUtils;
import adams.env.Environment;
import adams.test.AbstractTestHelper;
import adams.test.AdamsTestCase;
import adams.test.TestHelper;
import adams.test.TmpFile;

/**
 * Tests for the FileUtils utility class.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3741 $
 */
public class FileUtilsTest
  extends AdamsTestCase {

  /**
   * Initializes the test.
   *
   * @param name	the name of the test
   */
  public FileUtilsTest(String name) {
    super(name);
  }

  /**
   * Tests the quoteExcutable() methods.
   */
  public void testQuoteExecutable() {
    String filename = "/usr/bin/ls";
    assertEquals("No quotes expected", filename, FileUtils.quoteExecutable(filename));

    filename = "C:\\Program files\\Blah\\funky.exe";
    assertEquals("Quotes expected", "\"" + filename + "\"", FileUtils.quoteExecutable(filename));
  }

  /**
   * Tests the getExtension() methods.
   */
  public void testGetExtension() {
    String filename = "hello.txt";
    assertEquals("Extensions differ", "txt", FileUtils.getExtension(filename));
    filename = "hello.props";
    assertEquals("Extensions differ", "props", FileUtils.getExtension(filename));
    filename = "/some/where/hello.props";
    assertEquals("Extensions differ", "props", FileUtils.getExtension(filename));
    filename = "/some/where/hello";
    assertNull("No extension available", FileUtils.getExtension(filename));
    filename = "/some/where/hello.tar.gz";
    assertEquals("Extensions differ", "tar.gz", FileUtils.getExtension(filename));

    File file = new File("hello.txt");
    assertEquals("Extensions differ", "txt", FileUtils.getExtension(file));
    file = new File("hello.props");
    assertEquals("Extensions differ", "props", FileUtils.getExtension(file));
    file = new File("/some/where/hello.props");
    assertEquals("Extensions differ", "props", FileUtils.getExtension(file));
    file = new File("/some/where/hello");
    assertNull("No extension available", FileUtils.getExtension(file));
    file = new File("/some/where/hello.tar.gz");
    assertEquals("Extensions differ", "tar.gz", FileUtils.getExtension(file));
  }

  /**
   * Tests the getExtensions() methods.
   */
  public void testGetExtensions() {
    String filename = "hello.txt";
    assertEquals("# of Extensions differ", 1, FileUtils.getExtensions(filename).length);
    assertEquals("Extension differs", "txt", FileUtils.getExtensions(filename)[0]);
    filename = "/some/where/hello.tar.gz";
    assertEquals("# of Extensions differ", 2, FileUtils.getExtensions(filename).length);
    assertEquals("Extension differs", "tar.gz", FileUtils.getExtensions(filename)[0]);
    assertEquals("Extension differs", "gz", FileUtils.getExtensions(filename)[1]);

    File file = new File("hello.txt");
    assertEquals("# of Extensions differ", 1, FileUtils.getExtensions(file).length);
    assertEquals("Extension differs", "txt", FileUtils.getExtensions(file)[0]);
    file = new File("/some/where/hello.tar.gz");
    assertEquals("# of Extensions differ", 2, FileUtils.getExtensions(file).length);
    assertEquals("Extension differs", "tar.gz", FileUtils.getExtensions(file)[0]);
    assertEquals("Extension differs", "gz", FileUtils.getExtensions(file)[1]);
  }

  /**
   * Returns a test suite.
   *
   * @return		the test suite
   */
  public static Test suite() {
    return new TestSuite(FileUtilsTest.class);
  }

  /**
   * Runs the test from commandline.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    Environment.setEnvironmentClass(Environment.class);
    runTest(suite());
  }
}
