/**
 * Regression.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 * Copyright (C) Jeff Pace <jpace at incava dot org> (diff example class)
 */
package adams.test;

import java.io.File;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.incava.util.diff.Diff;
import org.incava.util.diff.Difference;

import adams.core.annotation.MixedCopyright;
import adams.core.io.FileUtils;

/**
 * Helper class for regression tests. Some of the code was taken from
 * Jeff Pace's <code>FileUtil.java</code> example class.
 * <p/>
 * Notes on the output:
 * <ul>
 *   <li>"&lt;" is from the reference output</li>
 *   <li>"&gt;" is from the current output</li>
 *   <li>"a" means "added"</li>
 *   <li>"d" means "deleted"</li>
 *   <li>"c" means "changed"</li>
 * </ul>
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @author Jeff Pace (jpace at incava dot org)
 * @version $Revision: 2319 $
 */
@MixedCopyright
public class Regression {

  /** the storage place for reference files. */
  public final static String REFERENCES = "src/test/resources/regression";

  /** the extension for references (incl dot). */
  public final static String EXTENSION = ".ref";

  /** the class this regression test is for. */
  protected Class m_RegressionClass;

  /** the regression file to use. */
  protected File m_ReferenceFile;

  /**
   * Initializes the regression check.
   *
   * @param cls		the class that is being checked
   */
  public Regression(Class cls) {
    super();

    m_RegressionClass = cls;
    m_ReferenceFile   = new File(REFERENCES + "/" + m_RegressionClass.getName().replace(".", "/") + EXTENSION);
  }

  /**
   * Returns the class this regression helper is for.
   *
   * @return		the class
   */
  public Class getRegressionClass() {
    return m_RegressionClass;
  }

  /**
   * Returns the file for storing the reference data in.
   *
   * @return		the reference file
   */
  public File getReferenceFile() {
    return m_ReferenceFile;
  }

  /**
   * Generates a 1-based string representation from the start and end position.
   *
   * @param start	the start index
   * @param end		the end index
   * @return		the generated string
   */
  protected String toString(int start, int end) {
    StringBuffer result;

    result = new StringBuffer();

    // match the line numbering from diff(1):
    result.append(end == Difference.NONE ? start : (1 + start));

    if (end != Difference.NONE && start != end)
      result.append(",").append(1 + end);

    return result.toString();
  }

  /**
   * Assembles the lines into a string.
   *
   * @param start	the start line index
   * @param end		the end line index (incl)
   * @param ind		the indicator string
   * @param lines	the underlying lines
   * @return		the generated string
   */
  protected String toString(int start, int end, String ind, List<String> lines) {
    StringBuffer	result;
    int			i;

    result = new StringBuffer();

    for (i = start; i <= end; i++)
      result.append(ind + " " + lines.get(i) + "\n");

    return result.toString();
  }

  /**
   * Compares the content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   *
   * @param content	the content to check
   * @return		the difference (or an error message), null if no difference
   */
  protected String compare(List<String> content) {
    StringBuilder	result;
    List<String>	reference;
    List		diffs;
    Iterator 		iter;
    Difference 		diff;
    int        		delStart;
    int        		delEnd;
    int        		addStart;
    int        		addEnd;
    String     		from;
    String     		to;
    String     		type;

    // reference available?
    if (!getReferenceFile().exists()) {
      if (!getReferenceFile().getParentFile().exists()) {
	if (!getReferenceFile().getParentFile().mkdirs())
	  return "Failed to create reference file: " + m_ReferenceFile;
      }

      if (FileUtils.saveToFile(content, getReferenceFile())) {
	System.err.println("Reference file created: " + m_ReferenceFile);
	return null;
      }
      else {
	return "Failed to create reference file: " + m_ReferenceFile;
      }
    }

    // compare
    reference = FileUtils.loadFromFile(getReferenceFile());
    diffs     = new Diff(reference, content).diff();
    result    = new StringBuilder();
    iter      = diffs.iterator();
    while (iter.hasNext()) {
      diff     = (Difference) iter.next();
      delStart = diff.getDeletedStart();
      delEnd   = diff.getDeletedEnd();
      addStart = diff.getAddedStart();
      addEnd   = diff.getAddedEnd();
      from     = toString(delStart, delEnd);
      to       = toString(addStart, addEnd);
      type     = delEnd != Difference.NONE && addEnd != Difference.NONE ? "c" : (delEnd == Difference.NONE ? "a" : "d");

      result.append(from + type + to + "\n");

      if (delEnd != Difference.NONE) {
	result.append(toString(delStart, delEnd, "<", reference));
	if (addEnd != Difference.NONE)
	  result.append("---\n");
      }
      if (addEnd != Difference.NONE) {
	result.append(toString(addStart, addEnd, ">", content));
      }
    }

    if (result.length() == 0)
      return null;
    else
      return result.toString();
  }

  /**
   * Trims the list, i.e., removes the ignored indices from it.
   *
   * @param list	the list to trim
   * @param ignore	the ignored indices
   * @return		the trimmed list
   */
  public static List<String> trim(List<String> list, int[] ignore) {
    Vector<String>	result;
    HashSet<Integer>	ignored;
    int			i;

    if (ignore.length == 0)
      return list;

    result  = new Vector<String>();
    ignored = new HashSet<Integer>();
    for (int ign: ignore)
      ignored.add(ign);

    for (i = 0; i < list.size(); i++) {
      if (ignored.contains(i))
	continue;
      result.add(list.get(i));
    }

    return result;
  }

  /**
   * Compares the content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   * Uses "\n" for splitting the string content into lines.
   *
   * @param content	the content to check
   * @return		the difference, null if no difference
   */
  public String compare(String content) {
    return compare(content, "\n");
  }

  /**
   * Compares the content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   * Uses "\n" for splitting the string content into lines.
   *
   * @param content	the content to check
   * @param ignore	the indices of the lines to ignore
   * @return		the difference, null if no difference
   */
  public String compare(String content, int[] ignore) {
    return compare(content, "\n", ignore);
  }

  /**
   * Compares the content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   *
   * @param content	the content to check
   * @param nl		the new line string to use
   * @return		the difference, null if no difference
   */
  public String compare(String content, String nl) {
    return compare(content, nl, new int[0]);
  }

  /**
   * Compares the content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   *
   * @param content	the content to check
   * @param nl		the new line string to use
   * @param ignore	the indices of the lines to ignore
   * @return		the difference, null if no difference
   */
  public String compare(String content, String nl, int[] ignore) {
    return compare(trim(Arrays.asList(content.split(nl)), ignore));
  }

  /**
   * Compares the file content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   *
   * @param files	the files containing the generated output to check
   * @return		the difference, null if no difference
   */
  public String compare(File[] files) {
    return compare(files, new int[0]);
  }

  /**
   * Compares the file content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   *
   * @param files	the files containing the generated output to check
   * @param ignore	the line indices in the files to ignore
   * @return		the difference, null if no difference
   */
  public String compare(File[] files, int[] ignore) {
    Vector<String>	content;
    int			i;

    content = new Vector<String>();
    for (i = 0; i < files.length; i++) {
      content.add("--> " + files[i].getName());
      if (!files[i].exists())
	content.add("file is missing");
      else
	content.addAll(trim(FileUtils.loadFromFile(files[i]), ignore));
      content.add("");
    }

    return compare(content);
  }

  /**
   * Compares the file content generated by the specified class with the stored
   * regression data. Creates a new reference file if not present yet.
   *
   * @param files	the files containing the generated output to check
   * @return		the difference, null if no difference
   */
  public String compareBinary(File[] files) {
    Vector<String>	content;
    int			i;

    content = new Vector<String>();
    for (i = 0; i < files.length; i++) {
      content.add("--> " + files[i].getName());
      if (!files[i].exists())
	content.add("file is missing");
      else
	content.add(new String(FileUtils.loadFromBinaryFile(files[i])));
      content.add("");
    }

    return compare(content);
  }
}
