/*
 * PrintObject.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.core;

import java.io.Serializable;

import adams.core.io.Log;
import adams.gui.core.ConsolePanel;

/**
 * A class that manages the printing to the streams (stdout/stderr).
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3610 $
 */
public class PrintObject
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = 8419141294301631442L;

  /**
   * The type of output to handle.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3610 $
   */
  public enum OutputType {
    /** stdout. */
    STDOUT,
    /** stderr. */
    STDERR,
    /** debug. */
    DEBUG
  }

  /** the owner's classname. */
  protected String m_OwnerClassname;

  /** the owner's hashcode. */
  protected int m_OwnerHashcode;

  /** the output type. */
  protected OutputType m_OutputType;

  /** the prefix of this object. */
  protected String m_Prefix;

  /** the suffix of this object. */
  protected String m_Suffix;

  /** whether stdout or stderr. */
  protected boolean m_UseStdOut;

  /** whether printing is enabled. */
  protected boolean m_Enabled;

  /** the console panel. */
  protected transient ConsolePanel m_ConsolePanel;

  /**
   * Initializes the print object.
   *
   * @param owner		the owner
   * @param outputType		the type of output to handle
   * @param enabled		whether the printing is enabled
   * @param prefix		the prefix to use (null for automatic generation)
   */
  public PrintObject(Object owner, OutputType outputType, boolean enabled, String prefix) {
    m_OwnerClassname = owner.getClass().getName();
    m_OwnerHashcode  = owner.hashCode();
    m_OutputType     = outputType;
    m_Prefix         = prefix;
    m_Enabled        = enabled;
    switch (m_OutputType) {
      case STDOUT:
	m_UseStdOut = true;
	m_Suffix    = "-OUT";
	break;
      case STDERR:
	m_UseStdOut = false;
	m_Suffix    = "-ERR";
	break;
      case DEBUG:
	m_UseStdOut = true;
	m_Suffix    = "-DEBUG";
	break;
      default:
	throw new IllegalArgumentException("Unhandled output type: " + m_OutputType);
    }
  }

  /**
   * Generates the prefix for println.
   *
   * @return		the generated prefix
   */
  public String createPrefix() {
    String	result;

    result =   m_OwnerClassname
             + "/"
             + m_OwnerHashcode;

    return result;
  }

  /**
   * Sets the prefix to use.
   *
   * @param value	the new prefix
   */
  public void setPrefix(String value) {
    m_Prefix = value;
  }

  /**
   * Returns the prefix.
   *
   * @return		the prefix
   */
  public String getPrefix() {
    if (m_Prefix == null)
      m_Prefix = createPrefix();

    return m_Prefix;
  }

  /**
   * Sets the suffix to use.
   *
   * @param value	the new suffix
   */
  public void setSuffix(String value) {
    m_Suffix = value;
  }

  /**
   * Returns the suffix.
   *
   * @return		the suffix
   */
  public String getSuffix() {
    return m_Suffix;
  }

  /**
   * Sets whether the printing is enabled or not.
   *
   * @param value	if true then the printing is enabled
   */
  public void setEnabled(boolean value) {
    m_Enabled = value;
  }

  /**
   * Returns whether the printing is enabled or not.
   *
   * @return		true if the printing is enabled
   */
  public boolean isEnabled() {
    return m_Enabled;
  }

  /**
   * Returns the initially set output type.
   *
   * @return		the output type
   */
  public OutputType getOutputType() {
    return m_OutputType;
  }

  /**
   * Prints the prefix.
   */
  public void printPrefix() {
    print(getPrefix());
  }

  /**
   * Prints the message using the Log.getSingleton() instance.
   * No prefix is used.
   *
   * @param msg		the message to print
   */
  protected void doPrint(String msg) {
    if (!m_Enabled)
      return;

    if (m_UseStdOut)
      Log.getSingleton().printOut(msg);
    else
      Log.getSingleton().printErr(msg);

    if (m_ConsolePanel == null)
      m_ConsolePanel = ConsolePanel.getSingleton();
    m_ConsolePanel.append(m_OutputType, msg);
  }

  /**
   * Prints the given string.
   *
   * @param msg		the string to print
   */
  public void print(String msg) {
    doPrint(msg);
  }

  /**
   * Prints the given boolean.
   *
   * @param x		the boolean to print
   */
  public void print(boolean x) {
    doPrint(Boolean.toString(x));
  }

  /**
   * Prints the given char.
   *
   * @param x		the char to print
   */
  public void print(char x) {
    doPrint(Character.toString(x));
  }

  /**
   * Prints the given chars.
   *
   * @param x		the chars to print
   */
  public void print(char[] x) {
    doPrint(new String(x));
  }

  /**
   * Prints the given double.
   *
   * @param x		the double to print
   */
  public void print(double x) {
    doPrint(Double.toString(x));
  }

  /**
   * Prints the given float.
   *
   * @param x		the float to print
   */
  public void print(float x) {
    doPrint(Float.toString(x));
  }

  /**
   * Prints the given int.
   *
   * @param x		the int to print
   */
  public void print(int x) {
    doPrint(Integer.toString(x));
  }

  /**
   * Prints the given object.
   *
   * @param x		the object to print
   */
  public void print(Object x) {
    doPrint("" + x);
  }

  /**
   * Prints the given long.
   *
   * @param x		the long to print
   */
  public void print(long x) {
    doPrint(Long.toString(x));
  }

  /**
   * Prints the message to the Log.getSingleton() instance. Each line in the
   * message gets prefix with classname (abbreviated) and hashcode of this
   * object plus the suffix: [&lt;prefix&gt;&lt;suffix&gt;].
   *
   * @param msg		the message to print
   */
  protected void doPrintln(String msg) {
    StringBuilder	msgBuf;
    String[]		lines;
    int		i;

    if (!m_Enabled)
      return;

    // for initializing
    getPrefix();

    msgBuf = new StringBuilder();
    lines  = msg.split("\n");
    for (i = 0; i < lines.length; i++) {
      if (i > 0)
        msgBuf.append("\n");
      msgBuf.append("[");
      msgBuf.append(m_Prefix);
      msgBuf.append(m_Suffix);
      msgBuf.append("] ");
      msgBuf.append(lines[i]);
    }

    if (m_UseStdOut)
      Log.getSingleton().printlnOut(msgBuf.toString());
    else
      Log.getSingleton().printlnErr(msgBuf.toString());

    msgBuf.append("\n");
    if (m_ConsolePanel == null)
      m_ConsolePanel = ConsolePanel.getSingleton();
    m_ConsolePanel.append(m_OutputType, msgBuf.toString());

    msgBuf = null;
    lines  = null;
  }

  /**
   * Prints the given string.
   *
   * @param msg		the string to print
   */
  public void println(String msg) {
    doPrintln(msg);
  }

  /**
   * Prints a new line.
   */
  public void println() {
    doPrintln("");
  }

  /**
   * Prints the given boolean.
   *
   * @param x		the boolean to print
   */
  public void println(boolean x) {
    doPrintln(Boolean.toString(x));
  }

  /**
   * Prints the given char.
   *
   * @param x		the char to print
   */
  public void println(char x) {
    doPrintln(Character.toString(x));
  }

  /**
   * Prints the given chars.
   *
   * @param x		the chars to print
   */
  public void println(char[] x) {
    doPrintln(new String(x));
  }

  /**
   * Prints the given double.
   *
   * @param x		the double to print
   */
  public void println(double x) {
    doPrintln(Double.toString(x));
  }

  /**
   * Prints the given float.
   *
   * @param x		the float to print
   */
  public void println(float x) {
    doPrintln(Float.toString(x));
  }

  /**
   * Prints the given int.
   *
   * @param x		the int to print
   */
  public void println(int x) {
    doPrintln(Integer.toString(x));
  }

  /**
   * Prints the given object.
   *
   * @param x		the object to print
   */
  public void println(Object x) {
    doPrintln("" + x);
  }

  /**
   * Prints the given long.
   *
   * @param x		the long to print
   */
  public void println(long x) {
    doPrintln(Long.toString(x));
  }

  /**
   * Prints the given throwable.
   *
   * @param x		the throwable to print
   */
  public void printStackTrace(Throwable x) {
    doPrintln(Utils.throwableToString(x));
  }
}