/*
 * BasePassword.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.core.base;

import org.ujmp.core.util.Base64;

/**
 * Wrapper for a String object to be editable in the GOE. Used for entering
 * passwords which get Base64 encoded.
 * <p/>
 * This not an attempt to keep passwords safe in any way, merely of obscuring
 * them slightly.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2961 $
 */
public class BasePassword
  extends BaseString {

  /** for serialization. */
  private static final long serialVersionUID = -3626665478868498359L;

  /** the indicator of base64 encoding (start). */
  public final static String BASE64_START = "{";

  /** the indicator of base64 encoding (end). */
  public final static String BASE64_END = "}";

  /** the error constant, if decoding failed. */
  public final static String ERROR = "-ERROR-";

  /**
   * Initializes the string with length 0.
   */
  public BasePassword() {
    this("");
  }

  /**
   * Initializes the object with the string to parse.
   *
   * @param s		the string to parse
   */
  public BasePassword(String s) {
    super(s);
  }

  /**
   * Turns the base64 encoded string into plain text.
   * Expects the string to be surrounded by BASE64_START and BASE64_END
   *
   * @param base64	the encoded string
   * @return		the decoded string, null in case of an error
   * @see		#BASE64_START
   * @see		#BASE64_END
   */
  protected String decode(String base64) {
    String	result;

    try {
      result = new String(Base64.decode(base64.substring(1, base64.length() - 1)));
    }
    catch (Exception e) {
      System.err.println("Failed to decode base64 string: " + base64);
      e.printStackTrace();
      result = ERROR;
    }

    return result;
  }

  /**
   * Encodes the given string in base64.
   *
   * @param raw		the string to encode
   * @return		the encoded string
   */
  protected String encode(String raw) {
    return BASE64_START + Base64.encodeBytes(raw.getBytes()) + BASE64_END;
  }

  /**
   * Checks whether the string value is a valid presentation for this class.
   *
   * @param value	the string value to check
   * @return		always true if not base64 encoded, otherwise the base64
   * 			string must be properly surrounded
   * @see		#BASE64_START
   * @see		#BASE64_END
   */
  public boolean isValid(String value) {
    if (value.startsWith(BASE64_START))
      return value.endsWith(BASE64_END);
    else
      return true;
  }

  /**
   * Sets the string value.
   *
   * @param value	the string value, clear text or base64 encoded
   */
  public void setValue(String value) {
    if (!isValid(value))
      return;

    if (value.startsWith(BASE64_START))
      m_Internal = decode(value);
    else
      m_Internal = value;
  }

  /**
   * Returns the current string value.
   *
   * @return		the string value, clear text
   */
  public String getValue() {
    return (String) m_Internal;
  }

  /**
   * Returns the backquoted String value.
   *
   * @return		the base64 encoded string value
   */
  public String stringValue() {
    return encode(getValue());
  }

  /**
   * Returns a tool tip for the GUI editor (ignored if null is returned).
   *
   * @return		the tool tip
   */
  public String getTipText() {
    return "For handling passwords.";
  }
}
