/**
 * AbstractSpreadSheetReader.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;

import org.apache.commons.io.input.ReaderInputStream;

import adams.core.ClassLister;
import adams.core.option.AbstractOptionHandler;

/**
 * Ancestor for classes that can read spreadsheets.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 */
public abstract class AbstractSpreadSheetReader
  extends AbstractOptionHandler {

  /** for serialization. */
  private static final long serialVersionUID = 4828477005893179066L;

  /** The placeholder for missing values. */
  protected String m_MissingValue;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "missing", "missingValue",
	    "");
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public abstract String getFormatDescription();

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public abstract String[] getFormatExtensions();

  /**
   * Sets the placeholder for missing values.
   *
   * @param value	the placeholder
   */
  public void setMissingValue(String value) {
    m_MissingValue = value;
  }

  /**
   * Returns the current placeholder for missing values.
   *
   * @return		the placeholder
   */
  public String getMissingValue() {
    return m_MissingValue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String missingValueTipText() {
    return "The placeholder for missing values.";
  }

  /**
   * Returns whether to read from an InputStream rather than a Reader when
   * using a file name.
   *
   * @return		true if to read from an InputStream
   */
  protected abstract boolean getUseInputStream();

  /**
   * Reads the spreadsheet from the specified file.
   *
   * @param file	the file to read from
   * @return		null in case of an error, otherwise the spreadsheet
   */
  public SpreadSheet read(File file) {
    return read(file.getAbsolutePath());
  }

  /**
   * Reads the spreadsheet from the given file.
   *
   * @param filename	the file to read from
   * @return		the spreadsheet or null in case of an error
   */
  public SpreadSheet read(String filename) {
    SpreadSheet		result;
    BufferedReader	reader;
    InputStream		input;

    reader = null;
    input  = null;
    try {
      if (getUseInputStream()) {
	input = new FileInputStream(filename);
	result = doRead(input);
      }
      else {
	reader = new BufferedReader(new FileReader(filename));
	result = doRead(reader);
      }
    }
    catch (Exception e) {
      result = null;
      e.printStackTrace();
    }
    finally {
      if (reader != null) {
	try {
	  reader.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
      if (input != null) {
	try {
	  input.close();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return result;
  }

  /**
   * Reads the spreadsheet from the stream. The caller must ensure to
   * close the stream.
   *
   * @param stream	the stream to read from
   * @return		the spreadsheet or null in case of an error
   */
  public SpreadSheet read(InputStream stream) {
    SpreadSheet		result;

    try {
      if (getUseInputStream())
	result = doRead(stream);
      else
	result = doRead(new BufferedReader(new InputStreamReader(stream)));
    }
    catch (Exception e) {
      result = null;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Reads the spreadsheet from the given reader. The caller must ensure to
   * close the reader.
   *
   * @param r		the reader to read from
   * @return		the spreadsheet or null in case of an error
   */
  public SpreadSheet read(Reader r) {
    SpreadSheet		result;

    try {
      if (getUseInputStream())
	result = doRead(new ReaderInputStream(r));
      else
	result = doRead(r);
    }
    catch (Exception e) {
      result = null;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Performs the actual reading.
   * <p/>
   * Default implementation returns null.
   *
   * @param r		the reader to read from
   * @return		the spreadsheet or null in case of an error
   * @see		#getUseInputStream()
   */
  protected SpreadSheet doRead(Reader r) {
    return null;
  }

  /**
   * Performs the actual reading.
   * <p/>
   * Default implementation returns null.
   *
   * @param in		the input stream to read from
   * @return		the spreadsheet or null in case of an error
   * @see		#getUseInputStream()
   */
  protected SpreadSheet doRead(InputStream in) {
    return null;
  }

  /**
   * Returns a list with classnames of readers.
   *
   * @return		the reader classnames
   */
  public static String[] getReaders() {
    return ClassLister.getSingleton().getClassnames(AbstractSpreadSheetReader.class);
  }
}
