/*
 * AbstractSimpleReportWriter.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.data.io.output;

import java.io.FileWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Vector;

import adams.core.Properties;
import adams.core.Utils;
import adams.data.io.input.AbstractSimpleReportReader;
import adams.data.report.AbstractField;
import adams.data.report.Report;

/**
 * Abstract ancestor for writing reports in properties format.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 * @param <T> the type of report to handle
 */
public abstract class AbstractSimpleReportWriter<T extends Report>
  extends AbstractReportWriter<T> {

  /** for serialization. */
  private static final long serialVersionUID = 1281189381638349284L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Writes reports in properties file format.";
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public String getFormatDescription() {
    return "Properties file format";
  }

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension(s) (without the dot!)
   */
  public String[] getFormatExtensions() {
    return new String[]{AbstractSimpleReportReader.FILE_EXTENSION};
  }

  /**
   * Performs the actual writing.
   *
   * @param data	the data to write
   * @return		true if successfully written
   */
  protected boolean writeData(T data) {
    boolean			result;
    Properties			props;
    FileWriter			writer;
    Vector<AbstractField>	fields;
    int				i;
    StringWriter		swriter;
    String[]			lines;

    props = new Properties();

    // the parent ID
    props.setInteger(Report.PROPERTY_PARENTID, data.getDatabaseID());

    // transfer properties
    fields = data.getFields();
    for (i = 0; i < fields.size(); i++) {
      props.setProperty(fields.get(i).toString(), data.getValue(fields.get(i)).toString());
      props.setProperty(fields.get(i).toString() + Report.DATATYPE_SUFFIX, fields.get(i).getDataType().toString());
    }

    // write props file
    try {
      swriter = new StringWriter();
      props.store(swriter, "Simple report format (= Java properties file format)");
      lines = swriter.toString().split("\n");
      Arrays.sort(lines);
      writer = new FileWriter(m_Output.getAbsolutePath());
      writer.write(Utils.flatten(lines, "\n"));
      writer.close();
      result = true;
    }
    catch (Exception e) {
      result = false;
      getSystemErr().printStackTrace(e);
    }

    return result;
  }
}
