/*
 * Sequence.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.Hashtable;

import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.InputConsumer;
import adams.flow.core.InstantiatableActor;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Encapsulates a sequence of flow items.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: Sequence
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseString&gt; [-annotation ...] (property: annotations)
 *         The annotations to attach to this actor.
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-progress (property: showProgress)
 *         If set to true, progress information will be output to stdout ('.').
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor [options]&gt; [-actor ...] (property: actors)
 *         All the actors that define this sequence.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4245 $
 */
public class Sequence
  extends MutableConnectedControlActor
  implements InputConsumer, InstantiatableActor {

  /** for serialization. */
  private static final long serialVersionUID = -9211041097478667239L;

  /** the key for storing the current token in the backup. */
  public final static String BACKUP_CURRENT = "current";

  /** the token that gets passed on to all sub-branches. */
  protected transient Token m_CurrentToken;

  /** whether to allow standalones or not. */
  protected boolean m_AllowStandalones;

  /** whether to allow a source or not. */
  protected boolean m_AllowSource;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Encapsulates a sequence of flow items.";
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorsTipText() {
    return "All the actors that define this sequence.";
  }

  /**
   * Sets whether standalones are allowed or not.
   *
   * @param value	true if standalones are allowed
   */
  public void setAllowStandalones(boolean value) {
    m_AllowStandalones = value;
  }

  /**
   * Returns whether standalones are allowed or not.
   *
   * @return		true if standalones are allowed
   */
  public boolean getAllowStandalones() {
    return m_AllowStandalones;
  }

  /**
   * Sets whether a source is allowed or not.
   *
   * @param value	true if a source is allowed
   */
  public void setAllowSource(boolean value) {
    m_AllowSource = value;
  }

  /**
   * Returns whether a source is allowed or not.
   *
   * @return		true if a source is allowed
   */
  public boolean getAllowSource() {
    return m_AllowSource;
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(m_AllowStandalones, m_AllowSource, ActorExecution.SEQUENTIAL, true);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_CurrentToken != null)
      result.put(BACKUP_CURRENT, m_CurrentToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_CURRENT)) {
      m_CurrentToken = (Token) state.get(BACKUP_CURRENT);
      input(m_CurrentToken);
      state.remove(BACKUP_CURRENT);
    }

    super.restoreState(state);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the Class of objects that can be processed
   */
  public Class[] accepts() {
    Class[]		result;
    AbstractActor	first;

    result = new Class[]{Unknown.class};

    first = firstActive();
    if ((first != null) && (first instanceof InputConsumer))
      result = ((InputConsumer) first).accepts();

    return result;
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
    AbstractActor	first;

    m_CurrentToken = token;

    first = firstActive();
    if (isDebugOn())
      debug("first active actor: " + first.getFullName());
    if ((first != null) && (first instanceof InputConsumer)) {
      ((InputConsumer) first).input(m_CurrentToken);
      if (getDebugLevel() > 1)
	debug("input token: " + m_CurrentToken, 2);
    }
  }

  /**
   * Post-execute hook.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String postExecute() {
    m_CurrentToken = null;
    return super.postExecute();
  }

  /**
   * Finishes up the execution.
   */
  public void wrapUp() {
    m_CurrentToken = null;

    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    m_CurrentToken = null;

    super.cleanUp();
  }
}
