/*
 * MoveFile.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.io.File;

import adams.core.io.FileUtils;
import adams.core.io.PlaceholderFile;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Moves a file to a different location (or just renames it).<br/>
 * Source and target can be swapped as well.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * <pre>   java.lang.String</pre>
 * <pre>   java.io.File</pre>
 * - generates:<br/>
 * <pre>   java.lang.String</pre>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: MoveFile
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 *         The annotations to attach to this actor.
 *         default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-file &lt;adams.core.io.PlaceholderFile&gt; (property: file)
 *         The target file.
 *         default: .
 * </pre>
 *
 * <pre>-input-is-target (property: inputIsTarget)
 *         If true, then the input token will be used as target and the file parameter
 *          as source.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3124 $
 */
public class MoveFile
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -1725398133887399010L;

  /** the target. */
  protected PlaceholderFile m_File;

  /** whether the input token is the target instead. */
  protected boolean m_InputIsTarget;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Moves a file to a different location (or just renames it).\n"
      + "Source and target can be swapped as well.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "file", "file",
	    new PlaceholderFile("."));

    m_OptionManager.add(
	    "input-is-target", "inputIsTarget",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("file");
    if (variable != null)
      result = variable;
    else
      result = "" + m_File;

    if (m_InputIsTarget)
      result += " (input is target)";

    return result;
  }

  /**
   * Sets the target file.
   *
   * @param value	the file
   */
  public void setFile(PlaceholderFile value) {
    m_File = value;
    reset();
  }

  /**
   * Returns the target file.
   *
   * @return 		the file
   */
  public PlaceholderFile getFile() {
    return m_File;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String fileTipText() {
    return "The target file.";
  }

  /**
   * Sets whether the input is the actual target, not the file.
   *
   * @param value	if true then the input will be used as target
   * 			and the file as source
   */
  public void setInputIsTarget(boolean value) {
    m_InputIsTarget = value;
    reset();
  }

  /**
   * Returns whether the input is the actual target, not the file.
   *
   * @return 		true if the input will be used as target and the
   * 			file as source
   */
  public boolean getInputIsTarget() {
    return m_InputIsTarget;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String inputIsTargetTipText() {
    return
        "If true, then the input token will be used as target and the file "
      + "parameter as source.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String.class, java.io.File.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String.class, File.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class, java.io.File.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    File	file;
    File	source;
    File	target;

    try {
      if (m_InputToken.getPayload() instanceof File)
	file = new PlaceholderFile((File) m_InputToken.getPayload());
      else
	file = new PlaceholderFile((String) m_InputToken.getPayload());

      if (m_InputIsTarget) {
	target = file;
	source = m_File;
      }
      else {
	target = m_File;
	source = file;
      }

      debug("Source '" + source + "' exists: " + source.exists());
      debug("Target '" + target + "' exists: " + target.exists());
      FileUtils.move(source.getAbsoluteFile(), target.getAbsoluteFile());

      result = null;
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    m_OutputToken = new Token(m_InputToken.getPayload());

    return result;
  }
}
