/*
 * ReportField.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.Hashtable;
import java.util.Vector;

import adams.core.base.BaseRegExp;
import adams.data.report.AbstractField;
import adams.data.report.Report;
import adams.data.report.ReportHandler;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Returns all the fields in a report or the report from a report handling object that match the provided regular expression.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.report.Report<br/>
 * &nbsp;&nbsp;&nbsp;adams.data.report.ReportHandler<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ReportField
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-regexp &lt;adams.core.base.BaseRegExp&gt; (property: regExp)
 * &nbsp;&nbsp;&nbsp;The regular expression to match the field names against.
 * &nbsp;&nbsp;&nbsp;default: .*
 * </pre>
 *
 * <pre>-output-name-only (property: outputNameOnly)
 * &nbsp;&nbsp;&nbsp;If enabled, only the field name is output and not the data type as well.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3518 $
 */
public class ReportField
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -2833759108269704357L;

  /** for backing up the fields. */
  public final static String BACKUP_FIELDS = "fields";

  /** the regular expression for the fields to match. */
  protected BaseRegExp m_RegExp;

  /** whether to output only the name (not the type). */
  protected boolean m_OutputNameOnly;

  /** the list of fields to forward. */
  protected Vector<AbstractField> m_Fields;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Returns all the fields in a report or the report from a report "
      + "handling object that match the provided regular expression.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "regexp", "regExp",
	    new BaseRegExp(BaseRegExp.MATCH_ALL));

    m_OptionManager.add(
	    "output-name-only", "outputNameOnly",
	    false);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Fields = new Vector<AbstractField>();
  }

  /**
   * Resets the actor.
   */
  protected void reset() {
    super.reset();

    m_Fields.clear();
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_FIELDS);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_FIELDS, m_Fields);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_FIELDS)) {
      m_Fields = (Vector<AbstractField>) state.get(BACKUP_FIELDS);
      state.remove(BACKUP_FIELDS);
    }

    super.restoreState(state);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("regExp");
    if (variable != null)
      result = variable;
    else
      result = m_RegExp.getValue();

    if (m_OutputNameOnly)
      result += " (name only)";
    else
      result += " (name + type)";

    return result;
  }

  /**
   * Sets the regular expressions to use.
   *
   * @param value	the regular expressions
   */
  public void setRegExp(BaseRegExp value) {
    m_RegExp = value;
    reset();
  }

  /**
   * Returns the regular expressions in use.
   *
   * @return 		the regular expressions
   */
  public BaseRegExp getRegExp() {
    return m_RegExp;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String regExpTipText() {
    return "The regular expression to match the field names against.";
  }

  /**
   * Sets whether to output only the name and not the type as well.
   *
   * @param value	if true only the name is output
   */
  public void setOutputNameOnly(boolean value) {
    m_OutputNameOnly = value;
    reset();
  }

  /**
   * Returns whether to output only the name and not the type as well.
   *
   * @return 		true if only the name is output
   */
  public boolean getOutputNameOnly() {
    return m_OutputNameOnly;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String outputNameOnlyTipText() {
    return "If enabled, only the field name is output and not the data type as well.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.data.report.Report.class, adams.data.report.ReportHandler.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Report.class, ReportHandler.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String			result;
    Report			report;
    Vector<AbstractField>	fields;

    try {
      m_Fields.clear();

      if (m_InputToken.getPayload() instanceof Report)
	report = (Report) m_InputToken.getPayload();
      else
	report = ((ReportHandler) m_InputToken.getPayload()).getReport();

      // match fields
      if (report != null) {
	fields = report.getFields();
	for (AbstractField field: fields) {
	  if (m_RegExp.isMatch(field.getName()))
	    m_Fields.add(field);
	}
	if (isDebugOn())
	  debug("Matching fields: " + m_Fields);
      }
      else {
	if (isDebugOn())
	  debug("No report available: " + m_InputToken);
      }

      result = null;
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_Fields.size() > 0);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    if (m_OutputNameOnly)
      result = new Token(m_Fields.get(0).toString());
    else
      result = new Token(m_Fields.get(0).toParseableString());
    m_Fields.remove(0);

    m_OutputToken = null;
    m_InputToken  = null;

    return result;
  }
}
