/**
 * AxisPanelOptions.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.core;

import adams.core.option.AbstractOptionHandler;
import adams.gui.visualization.core.axis.Type;
import adams.gui.visualization.core.plot.Axis;

/**
 <!-- globalinfo-start -->
 * Encapsulates options for an axis in a plot.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-label &lt;java.lang.String&gt; (property: label)
 * &nbsp;&nbsp;&nbsp;The label of the axis.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-type &lt;Absolute|Percent|Log10 absolute|Log10 percent|Log absolute|Log percent|Date|Time|Date&#47;time&gt; (property: type)
 * &nbsp;&nbsp;&nbsp;The type of the axis.
 * &nbsp;&nbsp;&nbsp;default: ABSOLUTE
 * </pre>
 *
 * <pre>-hide-grid-lines (property: showGridLines)
 * &nbsp;&nbsp;&nbsp;If enabled, grid lines are plotted as well.
 * </pre>
 *
 * <pre>-format &lt;java.lang.String&gt; (property: format)
 * &nbsp;&nbsp;&nbsp;The format for displaying the tick labels on the axis; current default:
 * &nbsp;&nbsp;&nbsp;0.00E0;-0.00E0
 * &nbsp;&nbsp;&nbsp;default: 0.00E0
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 */
public class AxisPanelOptions
  extends AbstractOptionHandler {

  /** for serialization. */
  private static final long serialVersionUID = 1675594412675760089L;

  /** the label of the axis. */
  protected String m_Label;

  /** the type of the axis. */
  protected Type m_Type;

  /** whether to show gridlines or not. */
  protected boolean m_ShowGridLines;

  /** whether to show only the major gridlines or not. */
  protected boolean m_ShowOnlyMajorGridLines;

  /** the number of ticks to use. */
  protected int m_NumTicks;

  /** the length in pixles of ticks to use. */
  protected int m_LengthTicks;

  /** every nth value to display. */
  protected int m_NthValueToShow;

  /** the custom number/date format. */
  protected String m_CustomFormat;

  /** the top margin. */
  protected double m_TopMargin;

  /** the bottom margin. */
  protected double m_BottomMargin;

  /** the width of the axis (for HORIZONTAL axes, this is the height, of course). */
  protected int m_Width;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Encapsulates options for an axis in a plot.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "label", "label",
	    "");

    m_OptionManager.add(
	    "type", "type",
	    Type.ABSOLUTE);

    m_OptionManager.add(
	    "hide-grid-lines", "showGridLines",
	    true);

    m_OptionManager.add(
	    "only-major-grid-lines", "showOnlyMajorGridLines",
	    false);

    m_OptionManager.add(
	    "num-ticks", "numTicks",
	    20, 1, null);

    m_OptionManager.add(
	    "length-ticks", "lengthTicks",
	    4, 1, null);

    m_OptionManager.add(
	    "nth-value", "nthValueToShow",
	    5, 0, null);

    m_OptionManager.add(
	    "width", "width",
	    20, 5, null);

    m_OptionManager.add(
	    "custom-format", "customFormat",
	    "");
  }

  /**
   * Sets label of the axis.
   *
   * @param value 	the label
   */
  public void setLabel(String value) {
    m_Label = value;
    reset();
  }

  /**
   * Returns the label of the axis.
   *
   * @return 		the label
   */
  public String getLabel() {
    return m_Label;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String labelTipText() {
    return "The label of the axis.";
  }

  /**
   * Sets type of the axis.
   *
   * @param value 	the type
   */
  public void setType(Type value) {
    m_Type = value;
    reset();
  }

  /**
   * Returns the type of the axis.
   *
   * @return 		the type
   */
  public Type getType() {
    return m_Type;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String typeTipText() {
    return "The type of the axis.";
  }

  /**
   * Sets whether to plot grid lines as well.
   *
   * @param value 	if true grid lines will be plotted
   */
  public void setShowGridLines(boolean value) {
    m_ShowGridLines = value;
    reset();
  }

  /**
   * Returns whether to plot grid lines as well.
   *
   * @return 		true if grid lines are plotted
   */
  public boolean getShowGridLines() {
    return m_ShowGridLines;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String showGridLinesTipText() {
    return "If enabled, grid lines are plotted as well.";
  }

  /**
   * Sets whether to plot only major grid lines as well.
   *
   * @param value 	if true only major grid lines will be plotted
   */
  public void setShowOnlyMajorGridLines(boolean value) {
    m_ShowOnlyMajorGridLines = value;
    reset();
  }

  /**
   * Returns whether to plot only major grid lines as well.
   *
   * @return 		true if only major grid lines are plotted
   */
  public boolean getShowOnlyMajorGridLines() {
    return m_ShowOnlyMajorGridLines;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String showOnlyMajorGridLinesTipText() {
    return "If enabled, only major grid lines are plotted.";
  }

  /**
   * Sets the count of ticks a value is shown, i.e., "3" means every third tick:
   * 1, 4, 7, ...
   *
   * @param value	the count
   */
  public void setNthValueToShow(int value) {
    if (m_NthValueToShow >= 0) {
      m_NthValueToShow = value;
      reset();
    }
    else {
      System.err.println("'n-th value to show' must be >=0, provided: " + value);
    }
  }

  /**
   * Returns the count of ticks a value is shown, i.e., "3" means every third
   * tick: 1, 4, 7, ...
   *
   * @return		the count
   */
  public int getNthValueToShow() {
    return m_NthValueToShow;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String nthValueToShowTipText() {
    return
        "Every n-th tick a value label is printed, eg, for '3' every third "
      + "tick would have a label: 1, 4, 7, etc.";
  }

  /**
   * Sets the number of ticks to display along the axis.
   *
   * @param value	the number of ticks
   */
  public void setNumTicks(int value) {
    if (value > 0) {
      m_NumTicks = value;
      reset();
    }
    else {
      System.err.println("Number of ticks must be >0, provided: " + value);
    }
  }

  /**
   * Returns the number of ticks currently displayed.
   *
   * @return		the number of ticks
   */
  public int getNumTicks() {
    return m_NumTicks;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String numTicksTipText() {
    return "The overall number of ticks to display.";
  }

  /**
   * Sets the length of ticks to display along the axis.
   *
   * @param value	the length of ticks (in pixels)
   */
  public void setLengthTicks(int value) {
    if (value > 0) {
      m_LengthTicks = value;
      reset();
    }
    else {
      System.err.println("Length of ticks must be >0, provided: " + value);
    }
  }

  /**
   * Returns the length of ticks currently displayed.
   *
   * @return		the length of ticks (in pixles)
   */
  public int getLengthTicks() {
    return m_LengthTicks;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String lengthTicksTipText() {
    return "The length in pixels of the ticks to display.";
  }

  /**
   * Sets the top margin factor (>= 0.0).
   *
   * @param value	the top margin
   */
  public void setTopMargin(double value) {
    if (value >= 0) {
      m_TopMargin = value;
      reset();
    }
    else {
      System.err.println(
	  "Top margin factor must be at least 0.0 (provided: " + value + ")!");
    }
  }

  /**
   * Returns the currently set top margin factor (>= 0.0).
   *
   * @return		the top margin
   */
  public double getTopMargin() {
    return m_TopMargin;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String topMarginTipText() {
    return "The factor for an extra margin on the top/left (eg 0.05 = 5%).";
  }

  /**
   * Sets the bottom margin factor (>= 0.0).
   *
   * @param value	the bottom margin
   */
  public void setBottomMargin(double value) {
    if (value >= 0) {
      m_BottomMargin = value;
      reset();
    }
    else {
      System.err.println(
	  "Bottom margin factor must be at least 0.0 (provided: " + value + ")!");
    }
  }

  /**
   * Returns the currently set bottom margin factor (>= 0.0).
   *
   * @return		the bottom margin
   */
  public double getBottomMargin() {
    return m_BottomMargin;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String bottomMarginTipText() {
    return "The factor for an extra margin on the bottom/right (eg 0.05 = 5%).";
  }

  /**
   * Sets the width of the axis (this is height for HORIZONTAL axes, of
   * course), at least 5 pixel.
   *
   * @param value	the new width
   */
  public void setWidth(int value) {
    if (value >= 5) {
      m_Width = value;
      reset();
    }
    else {
      System.err.println(
	  "The width must be at least 5 pixels (provided: " + value + ")!");
    }
  }

  /**
   * Returns the current width of the axis.
   *
   * @return		the width
   */
  public int getWidth() {
    return m_Width;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String widthTipText() {
    return "The width/height of the axis (>= 5).";
  }

  /**
   * Sets the custom format for the tick labels.
   *
   * @param value 	the custom format
   */
  public void setCustomFormat(String value) {
    m_CustomFormat = value;
    reset();
  }

  /**
   * Returns the custom format for the tick labels.
   *
   * @return 		the custom format
   */
  public String getCustomFormat() {
    return m_CustomFormat;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String customFormatTipText() {
    return
        "The custom format for displaying the tick labels on the axis.";
  }

  /**
   * Applies the options to the specified axis.
   *
   * @param plot	the plot panel to update an axis for
   * @param axis	the axis to configure
   */
  public void configure(PlotPanel plot, Axis axis) {
    plot.setAxisWidth(axis, m_Width);
    plot.getAxis(axis).setType(m_Type);
    plot.getAxis(axis).setShowGridLines(m_ShowGridLines);
    plot.getAxis(axis).setShowOnlyMajorGridLines(m_ShowOnlyMajorGridLines);
    plot.getAxis(axis).setNumTicks(m_NumTicks);
    plot.getAxis(axis).setLengthTicks(m_LengthTicks);
    plot.getAxis(axis).setNthValueToShow(m_NthValueToShow);
    plot.getAxis(axis).setTopMargin(m_TopMargin);
    plot.getAxis(axis).setBottomMargin(m_BottomMargin);
    plot.getAxis(axis).setAxisName(m_Label);
    if (m_CustomFormat.trim().length() > 0)
      plot.getAxis(axis).setNumberFormat(m_CustomFormat);
  }
}
