/*
 * BaseTime.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.core.base;

import java.util.Date;

import adams.core.DateFormat;
import adams.parser.BaseTimeExpression;
import adams.parser.GrammarSupplier;

/**
 * Wrapper for a Time string to be editable in the GOE. Dates have to be of
 * format "HH:mm:ss".
 * <pre>
 * parses expressions as follows:
 *   (&lt;date&gt;|NOW|-INF|+INF|START|END) [(+&lt;int&gt;|-&lt;int&gt;) (SECOND|MINUTE|HOUR)]
 * Examples:
 *   01:02:03
 *   01:02:03 +1 MINUTE
 *   NOW
 *   +INF
 *   NOW +1 HOUR
 *   NOW +14 MINNUTE
 * Amounts can be chained as well:
 *   NOW -1 MINUTE +1 HOUR
 * START and END can only be set programmatically; by default they equal to -INF and +INF.
 * </pre>
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3038 $
 * @see #DATE_FORMAT
 * @see #setStart(Date)
 * @see #setEnd(Date)
 */
public class BaseTime
  extends BaseObject
  implements GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = -5853830144343397434L;

  /** the placeholder for "-INF" (infinity in the past). */
  public final static String INF_PAST = "-INF";

  /** the "-INF" date. */
  public final static String INF_PAST_DATE = "00:00:00";

  /** the placeholder for "+INF" (infinity in the future). */
  public final static String INF_FUTURE = "+INF";

  /** the "+INF" date. */
  public final static String INF_FUTURE_DATE = "23:59:59";

  /** the placeholder for "now". */
  public final static String NOW = "NOW";

  /** the placeholder for "start". */
  public final static String START = "START";

  /** the placeholder for "end". */
  public final static String END = "END";

  /** the date format. */
  public final static String FORMAT = "HH:mm:ss";

  /** for formatting/parsing the dates. */
  protected static DateFormat m_Format;
  static {
    m_Format = new DateFormat(FORMAT);
  }

  /** the start time to use. */
  protected Date m_Start = null;

  /** the end time to use. */
  protected Date m_End = null;

  /**
   * Initializes the date as NOW.
   *
   * @see	#NOW
   */
  public BaseTime() {
    this(NOW);
  }

  /**
   * Initializes the object with the date to parse.
   *
   * @param s		the date to parse
   */
  public BaseTime(String s) {
    super(s);
  }

  /**
   * Initializes the object with the specified date.
   *
   * @param date	the date to use
   */
  public BaseTime(Date date) {
    this(m_Format.format(date));
  }

  /**
   * Sets the optional start time.
   *
   * @param value 	the start time
   */
  public void setStart(Date value) {
    m_Start = value;
  }

  /**
   * Returns the optional start time.
   *
   * @return 		the start time
   */
  public Date getStart() {
    return m_Start;
  }

  /**
   * Sets the optional end time.
   *
   * @param value 	the end time
   */
  public void setEnd(Date value) {
    m_End = value;
  }

  /**
   * Returns the optional end time.
   *
   * @return 		the end time
   */
  public Date getEnd() {
    return m_End;
  }

  /**
   * Parses the given date string.
   *
   * @param s		the date string to parse
   * @param quiet	whether to print exceptions or not
   * @return		the parsed date, null in case of an error
   */
  protected Date parse(String s, boolean quiet) {
    try {
      return BaseTimeExpression.evaluate(s, m_Start, m_End);
    }
    catch (Exception e) {
      if (!quiet) {
	System.err.println("Failed to parse: " + s);
	e.printStackTrace();
      }
      return null;
    }
  }

  /**
   * Checks whether the string value is a valid presentation for this class.
   *
   * @param value	the string value to check
   * @return		true if the string can be parsed
   */
  public boolean isValid(String value) {
    value = value.toUpperCase();

    if (value.length() == 0)
      return true;

    return (parse(value, true) != null);
  }

  /**
   * Sets the string value.
   *
   * @param value	the string value
   */
  public void setValue(String value) {
    if (!isValid(value))
      return;

    if (value.equals(INF_FUTURE_DATE))
      m_Internal = INF_FUTURE;
    else if (value.equals(INF_PAST_DATE))
      m_Internal = INF_PAST;
    else if (value.length() == 0)
      m_Internal = NOW;
    else
      m_Internal = value;
  }

  /**
   * Returns the current string value.
   *
   * @return		the string value
   */
  public String getValue() {
    return (String) m_Internal;
  }

  /**
   * Returns the Date value.
   *
   * @return		the Date value
   */
  public Date dateValue() {
    return parse(getValue(), false);
  }

  /**
   * Returns the actual Date as string.
   *
   * @return		the actual date as string
   */
  public String stringValue() {
    return m_Format.format(dateValue());
  }

  /**
   * Returns a tool tip for the GUI editor (ignored if null is returned).
   *
   * @return		the tool tip
   */
  public String getTipText() {
    return
        "A date of format '" + FORMAT + "' "
      + "("
      + "'" + INF_PAST + "' = '" + INF_PAST_DATE + "', "
      + "'" + INF_FUTURE + "' = '" + INF_FUTURE_DATE + "', "
      + "'" + NOW + "' = the current date/time"
      + ").";
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new BaseTimeExpression().getGrammar();
  }

  /**
   * Checks whether the date/time is +INF.
   *
   * @return		true if infinity future
   */
  public boolean isInfinityFuture() {
    return getValue().equals(INF_FUTURE);
  }

  /**
   * Checks whether the date/time is -INF.
   *
   * @return		true if infinity future
   */
  public boolean isInfinityPast() {
    return getValue().equals(INF_PAST);
  }

  /**
   * Checks whether the date/time is -INF or +INF.
   *
   * @return		true if any infinity
   */
  public boolean isInfinity() {
    return isInfinityPast() || isInfinityFuture();
  }

  /**
   * Returns a new BaseDate object initialized with the NOW placeholder.
   *
   * @return		the BaseDate object
   * @see		#NOW
   */
  public static BaseDateTime now() {
    return new BaseDateTime(NOW);
  }

  /**
   * Returns a new BaseDate object initialized with the INF_FUTURE placeholder.
   *
   * @return		the BaseDate object
   * @see		#INF_FUTURE
   */
  public static BaseDateTime infinityFuture() {
    return new BaseDateTime(INF_FUTURE);
  }

  /**
   * Returns a new BaseDate object initialized with the INF_PAST placeholder.
   *
   * @return		the BaseDate object
   * @see		#INF_PAST
   */
  public static BaseDateTime infinityPast() {
    return new BaseDateTime(INF_PAST);
  }
}
