/**
 * ClassOption.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Method;

/**
 * Option class for options with custom hooks for valueOf and toString.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2474 $
 * @see AbstractOption#m_HooksValueOf
 * @see AbstractOption#m_HooksToString
 */
public class CustomHooksOption
  extends AbstractArgumentOption {

  /** for serialization. */
  private static final long serialVersionUID = -3175126088440187209L;

  /**
   * Initializes the option. Will always output the default value.
   *
   * @param owner		the owner of this option
   * @param commandline		the commandline string to identify the option
   * @param property 		the name of bean property
   * @param defValue		the default value, if null then the owner's
   * 				current state is used
   */
  protected CustomHooksOption(OptionManager owner, String commandline, String property,
      Object defValue) {

    super(owner, commandline, property, defValue);
  }

  /**
   * Initializes the option.
   *
   * @param owner		the owner of this option
   * @param commandline		the commandline string to identify the option
   * @param property 		the name of bean property
   * @param defValue		the default value, if null then the owner's
   * 				current state is used
   * @param outputDefValue	whether to output the default value or not
   */
  protected CustomHooksOption(OptionManager owner, String commandline, String property,
      Object defValue, boolean outputDefValue) {

    super(owner, commandline, property, defValue, outputDefValue);
  }

  /**
   * Compares the two values.
   *
   * @param value	the value to compare against the default value
   * @param defValue	the default value to compare against
   * @return		true if both are equal
   */
  protected boolean compareValues(Object value, Object defValue) {
    return toString(value).equals(toString(defValue));
  }

  /**
   * Turns the string into the appropriate object.
   * <p/>
   * Needs to be overridden if no custom hook available.
   *
   * @param s		the string to parse
   * @return		the generated object
   * @throws Exception	if parsing of string fails
   */
  public Object valueOf(String s) throws Exception {
    Object	result;
    Method	method;

    method = OptionUtils.getValueOfHook(getBaseClass());
    if (method == null)
      throw new IllegalStateException("No 'valueOf' method defined for base class '" + getBaseClass() + "'?");
    else
      result = method.invoke(getOptionHandler(), new Object[]{this, s});

    return result;
  }

  /**
   * Returns a string representation of the specified object.
   * <p/>
   * Needs to be overridden if no custom hook available.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toString(Object obj) {
    String	result;
    Method	method;

    result = "";

    method = OptionUtils.getToStringHook(getBaseClass());
    if (method == null) {
      System.err.println("No 'toString' method defined for base class '" + getBaseClass() + "'?");
    }
    else {
      try {
	result = (String) method.invoke(getOptionHandler(), new Object[]{this, obj});
      }
      catch (Exception e) {
	System.err.println("Error obtaining string representation for '" + getProperty() + "':");
	e.printStackTrace();
      }
    }

    return result;
  }
}
