/**
 * AbstractRandomNumberGenerator.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.data.random;

import adams.core.ClassLister;
import adams.core.ShallowCopySupporter;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.AbstractOptionHandler;
import adams.core.option.OptionUtils;

/**
 * Ancestor for random number generators.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 * @param <T> the type of random number to return
 */
public abstract class AbstractRandomNumberGenerator<T extends Number>
  extends AbstractOptionHandler
  implements ShallowCopySupporter<AbstractRandomNumberGenerator> {

  /** for serialization. */
  private static final long serialVersionUID = 5803268124112742362L;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();
  }

  /**
   * Resets the generator.
   */
  public void reset() {
    super.reset();
  }

  /**
   * Performs optional checks.
   * <p/>
   * Default implementation does nothing.
   */
  protected void check() {
  }

  /**
   * Returns the next random number. Does the actual computation.
   *
   * @return		the next number
   */
  protected abstract T doNext();

  /**
   * Returns the nexct random number.
   *
   * @return		the next number
   */
  public synchronized T next() {
    check();
    return doNext();
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @return		the shallow copy
   */
  public AbstractRandomNumberGenerator shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself, i.e., based on the commandline options.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractRandomNumberGenerator shallowCopy(boolean expand) {
    return (AbstractRandomNumberGenerator) OptionUtils.shallowCopy(this, expand);
  }

  /**
   * Returns a list with classnames of statistics.
   *
   * @return		the statistic classnames
   */
  public static String[] getGenerators() {
    return ClassLister.getSingleton().getClassnames(AbstractRandomNumberGenerator.class);
  }

  /**
   * Instantiates the statistic with the given options.
   *
   * @param classname	the classname of the statistic to instantiate
   * @param options	the options for the statistic
   * @return		the instantiated statistic or null if an error occurred
   */
  public static AbstractRandomNumberGenerator forName(String classname, String[] options) {
    AbstractRandomNumberGenerator	result;

    try {
      result = (AbstractRandomNumberGenerator) OptionUtils.forName(AbstractRandomNumberGenerator.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the statistic from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			statistic to instantiate
   * @return		the instantiated statistic
   * 			or null if an error occurred
   */
  public static AbstractRandomNumberGenerator forCommandLine(String cmdline) {
    return (AbstractRandomNumberGenerator) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }
}
