/**
 * BooleanReportValue.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.condition.bool;

import adams.data.report.Field;
import adams.data.report.Report;
import adams.data.report.ReportHandler;
import adams.flow.core.AbstractActor;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Emulates an If-Then-Else construct. The 'Then' branch gets executed if the boolean field in the report is present and is 'true'.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-field &lt;adams.data.report.Field&gt; (property: field)
 * &nbsp;&nbsp;&nbsp;The field that determines the result of the evaluation.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4392 $
 */
public class BooleanReportValue
  extends AbstractBooleanCondition {

  /** for serialization. */
  private static final long serialVersionUID = -9169161144858552052L;

  /** the field in the report to check. */
  protected Field m_Field;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Emulates an If-Then-Else construct. The 'Then' branch gets executed "
      + "if the boolean field in the report is present and is 'true'.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "field", "field",
	    new Field());
  }

  /**
   * Sets the field to look for in the report.
   *
   * @param value	the field
   */
  public void setField(Field value) {
    m_Field = value;
    reset();
  }

  /**
   * Returns the field to look for in the report.
   *
   * @return		the field
   */
  public Field getField() {
    return m_Field;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String fieldTipText() {
    return "The field that determines the result of the evaluation.";
  }

  /**
   * Returns the quick info string to be displayed in the flow editor.
   *
   * @return		the info or null if no info to be displayed
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = "boolean report value: ";

    variable = getOptionManager().getVariableForProperty("field");
    if (variable != null)
      result += variable;
    else
      result += m_Field.toString();

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		ReportHandler, Report
   */
  public Class[] accepts() {
    return new Class[]{ReportHandler.class, Report.class};
  }

  /**
   * Configures the condition.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = null;

    if (m_Field == null)
      result = "No report field provided!";

    return result;
  }

  /**
   * Evaluates whether to executed the "then" or "else" branch.
   *
   * @param owner	the owning IfThenElse actor
   * @param token	the current token passing through the IfThenElse actor
   * @return		true if to execute the "then" branch
   */
  protected boolean doEvaluate(AbstractActor owner, Token token) {
    boolean	result;
    Report	report;

    result = false;

    if (token.getPayload() instanceof ReportHandler)
      report = ((ReportHandler) token.getPayload()).getReport();
    else
      report = (Report) token.getPayload();

    if (report.hasValue(m_Field))
      result = report.getBooleanValue(m_Field);

    return result;
  }
}
