/*
 *    AbstractScriptCondition.java
 *    Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.flow.condition.test;

import adams.core.io.PlaceholderFile;

/**
 * Abstract ancestor for actors that execute scripts.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4391 $
 */
public abstract class AbstractScriptCondition
  extends AbstractCondition {

  /** for serialization. */
  private static final long serialVersionUID = -1266048092842841686L;

  /** the Groovy module. */
  protected PlaceholderFile m_ScriptFile;

  /** the options for the Groovy module. */
  protected String m_ScriptOptions;

  /** the loaded script object. */
  protected transient Object m_ScriptObject;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "script", "scriptFile",
	    new PlaceholderFile("."));

    m_OptionManager.add(
	    "options", "scriptOptions",
	    "");
  }

  /**
   * Resets the condition.
   * Derived classes must call this method in set-methods of parameters to
   * assure the invalidation of previously generated data.
   */
  public void reset() {
    super.reset();

    m_ScriptObject = null;
  }

  /**
   * Sets the Groovy module.
   *
   * @param value 	the Groovy module
   */
  public void setScriptFile(PlaceholderFile value) {
    m_ScriptFile = value;
  }

  /**
   * Gets the Groovy module.
   *
   * @return 		the Groovy module
   */
  public PlaceholderFile getScriptFile() {
    return m_ScriptFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the explorer/experimenter gui
   */
  public String scriptFileTipText() {
    return "The script file to load and execute.";
  }

  /**
   * Sets the script options.
   *
   * @param value 	the options
   */
  public void setScriptOptions(String value) {
    m_ScriptOptions = value;
  }

  /**
   * Gets the script options.
   *
   * @return 		the options
   */
  public String getScriptOptions() {
    return m_ScriptOptions;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   * 			displaying in the explorer/experimenter gui
   */
  public String scriptOptionsTipText() {
    return "The options for the script.";
  }

  /**
   * Loads the scripts object and sets its options.
   *
   * @return		null if OK, otherwise the error message
   */
  protected abstract String loadScriptObject();

  /**
   * Checks the script object.
   *
   * @return		null if OK, otherwise the error message
   */
  protected abstract String checkScriptObject();

  /**
   * Tries to initialize the scripts object, sets its options and performs
   * some checks.
   *
   * @return		null if OK, otherwise the error message
   */
  protected String initScriptObject() {
    String	result;

    result = loadScriptObject();
    if (result == null)
      result = checkScriptObject();

    return result;
  }

  /**
   * Runs the script test.
   *
   * @return		the test result, null if everything OK, otherwise
   * 			the error message
   */
  protected abstract String performScriptTest();

  /**
   * Performs the actual testing of the condition.
   *
   * @return		the test result, null if everything OK, otherwise
   * 			the error message
   */
  protected String performTest() {
    String	result;

    result = null;

    if (m_ScriptObject == null)
      result = initScriptObject();

    if (result == null)
      result = performScriptTest();

    return result;
  }
}
