/**
 * RemoveUnusedGlobalActors.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.processor;

import java.util.HashSet;

import adams.flow.core.AbstractActor;
import adams.flow.core.AbstractGlobalActor;
import adams.flow.core.ActorHandler;
import adams.flow.core.ActorUtils;
import adams.flow.core.GlobalActorHelper;
import adams.flow.core.MutableActorHandler;
import adams.flow.standalone.GlobalActors;

/**
 <!-- globalinfo-start -->
 * Removes all unused global actors. Disabled actors referencing global actors are treated as if they were enabled. If a GlobalActors actor ends up being empty, it will get removed as well.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3569 $
 */
public class RemoveUnusedGlobalActors
  extends AbstractModifyingProcessor {

  /** for serialization. */

  /** blah. */
  private static final long serialVersionUID = 1634101991639994065L;
  /** the global actor helper. */
  protected GlobalActorHelper m_Helper;

  /**
   * Returns a string describing the object.
   *
   * @return 		a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Removes all unused global actors. Disabled actors referencing global "
      + "actors are treated as if they were enabled. If a GlobalActors actor "
      + "ends up being empty, it will get removed as well.";
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Helper = new GlobalActorHelper();
  }

  /**
   * Locates all global actors and returns their full names.
   *
   * @param actor	the actor to search below
   * @return		the full names
   */
  protected void locateGlobalActors(AbstractActor actor, HashSet<String> fullNames) {
    ActorHandler	handler;
    int			i;

    if (actor instanceof ActorHandler) {
      handler = (ActorHandler) actor;
      if (handler instanceof GlobalActors) {
	for (i = 0; i < handler.size(); i++)
	  fullNames.add(handler.get(i).getFullName());
      }
      else {
	for (i = 0; i < handler.size(); i++)
	  locateGlobalActors(handler.get(i), fullNames);
      }
    }
  }

  /**
   * Locates all references to global actors and returns their full names.
   *
   * @param actor	the actor to search below
   * @return		the full names of the global actors
   */
  protected void locateGlobalActorReferences(AbstractActor actor, HashSet<String> fullNames) {
    ActorHandler	handler;
    int			i;
    AbstractGlobalActor	reference;
    AbstractActor	global;

    if (actor instanceof AbstractGlobalActor) {
      reference = (AbstractGlobalActor) actor;
      global    = m_Helper.findGlobalActorRecursive(reference, reference.getGlobalName());
      if (global != null)
	fullNames.add(global.getFullName());
    }
    else if (actor instanceof ActorHandler) {
      handler = (ActorHandler) actor;
      for (i = 0; i < handler.size(); i++)
	locateGlobalActorReferences(handler.get(i), fullNames);
    }
  }

  /**
   * Removes the unused global actors.
   *
   * @param actor	the actor process
   * @param unused	the unused global actors to remove
   */
  protected void removeUnused(AbstractActor actor, HashSet<String> unused) {
    GlobalActors		globalActors;
    HashSet<GlobalActors>	parents;
    AbstractActor		global;

    m_ModifiedActor = actor.shallowCopy();
    m_Modified      = true;

    parents = new HashSet<GlobalActors>();
    for (String fullname: unused) {
      global = ActorUtils.locate(fullname, actor);
      if (global == null) {
	getSystemErr().println("Failed to locate global actor: " + fullname);
	continue;
      }
      globalActors = (GlobalActors) global.getParent();
      globalActors.remove(global.index());
      parents.add(globalActors);
    }

    // check for empty GlobalActors
    for (GlobalActors gactors: parents) {
      if (gactors.size() == 0) {
	if (isDebugOn())
	  debug("Removing empty GlobalActors: " + gactors.getFullName());
	((MutableActorHandler) gactors.getParent()).remove(gactors.index());
      }
    }
  }

  /**
   * Performs the actual processing.
   *
   * @param actor	the actor to process
   * @return		the processed actor
   */
  protected void processActor(AbstractActor actor) {
    HashSet<String>	global;
    HashSet<String>	referenced;
    HashSet<String>	unused;

    global = new HashSet<String>();
    locateGlobalActors(actor, global);
    if (isDebugOn())
      debug("global actors found: " + global);

    referenced = new HashSet<String>();
    locateGlobalActorReferences(actor, referenced);
    if (isDebugOn())
      debug("references found: " + referenced);

    unused = (HashSet<String>) global.clone();
    unused.removeAll(referenced);
    if (isDebugOn())
      debug("unused global actors: " + unused);

    if (unused.size() > 0)
      removeUnused(actor, unused);
  }
}
