/*
 * AbstractDatabaseConnection.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.standalone;

import adams.core.base.BasePassword;

/**
 * Ancestor for standalone actors providing a database connection different
 * from the system-wide one.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3750 $
 */
public abstract class AbstractDatabaseConnection
  extends AbstractStandalone {

  /** for serialization. */
  private static final long serialVersionUID = -1726172998200420556L;

  /** the URL to connect to the database. */
  protected String m_URL;

  /** database username. */
  protected String m_User;

  /** database password. */
  protected BasePassword m_Password;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "url", "URL",
	    "jdbc:mysql://somehost:3306/somedatabase", false);

    m_OptionManager.add(
	    "user", "user",
	    "", false);

    m_OptionManager.add(
	    "password", "password",
	    new BasePassword(), false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("URL");
    if (variable != null)
      result = variable;
    else
      result = getURL();

    return result;
  }

  /**
   * Sets the database URL.
   *
   * @param value	the URL
   */
  public void setURL(String value) {
    m_URL = value;
    reset();
  }

  /**
   * Returns the database URL.
   *
   * @return 		the URL
   */
  public String getURL() {
    return m_URL;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String URLTipText() {
    return "The JDBC URL of the database to connect to.";
  }

  /**
   * Sets the database user.
   *
   * @param value	the user
   */
  public void setUser(String value) {
    m_User = value;
    reset();
  }

  /**
   * Returns the database user.
   *
   * @return 		the user
   */
  public String getUser() {
    return m_User;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String userTipText() {
    return "The database user to connect with.";
  }

  /**
   * Sets the database password.
   *
   * @param value	the password
   */
  public void setPassword(BasePassword value) {
    m_Password = value;
    reset();
  }

  /**
   * Returns the database password.
   *
   * @return 		the password
   */
  public BasePassword getPassword() {
    return m_Password;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return         tip text for this property suitable for
   *             displaying in the GUI or for listing the options.
   */
  public String passwordTipText() {
    return "The password of the database user.";
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String				result;
    String				msg;
    adams.db.AbstractDatabaseConnection	conn;

    result = super.setUp();

    if (result == null) {
      conn = getConnection();
      msg  = null;
      if (!conn.isConnected() && !conn.getConnectOnStartUp()) {
	try {
	  conn.connect();
	}
	catch (Exception e) {
	  getSystemErr().println("Failed to connect to database (" + getURL() + "):");
	  getSystemErr().printStackTrace(e);
	  msg = e.toString();
	}
      }
      if (!conn.isConnected()) {
	result = "Failed to connect to database (" + getURL() + ")";
	if (msg == null)
	  result += "!";
	else
	  result += ": " + msg;
      }
    }

    return result;
  }

  /**
   * Executes the actor.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    return null;
  }

  /**
   * Returns the database connection in use. Reconnects the database, to make
   * sure that the database connection is the correct one.
   *
   * @return		the connection object
   */
  public abstract adams.db.AbstractDatabaseConnection getConnection();
}
