/*
 * ChromatogramDbReader.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.data.container.DataContainer;
import adams.db.AbstractDatabaseConnection;
import adams.db.DataProvider;
import adams.db.FilteredDataProvider;
import adams.flow.core.Token;
import adams.flow.provenance.ActorType;
import adams.flow.provenance.Provenance;
import adams.flow.provenance.ProvenanceContainer;
import adams.flow.provenance.ProvenanceInformation;
import adams.flow.provenance.ProvenanceSupporter;

/**
 <!-- globalinfo-start -->
 * Loads a chromatogram from the database and passes it on.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * <pre>   java.lang.Integer</pre>
 * - generates:<br/>
 * <pre>   gcms.data.chromatogram.Chromatogram</pre>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: ChromatogramDbReader
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseString&gt; [-annotation ...] (property: annotations)
 *         The annotations to attach to this actor.
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3507 $
 * @param <T> the type of data to load from the database
 */
public abstract class AbstractDataContainerDbReader<T extends DataContainer>
  extends AbstractTransformer
  implements ProvenanceSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -4736058667429890220L;

  /** whether to return the raw data or not. */
  protected boolean m_Raw;

  /** the database connection. */
  protected AbstractDatabaseConnection m_DatabaseConnection;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    // "raw" option is only available if connected
    if (getDataProvider() instanceof FilteredDataProvider) {
      m_OptionManager.add(
	  "raw", "raw",
	  false);
    }
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_DatabaseConnection = getDefaultDatabaseConnection();
  }

  /**
   * Returns the default database connection.
   *
   * @return 		the default database connection
   */
  protected abstract AbstractDatabaseConnection getDefaultDatabaseConnection();

  /**
   * Sets whether to return the raw data or not (only FilteredDataProviders).
   *
   * @param value 	true if transformation is to be skipped
   * @see		FilteredDataProvider
   */
  public void setRaw(boolean value) {
    m_Raw = value;
    reset();
  }

  /**
   * Returns whether to return the raw data or not (only FilteredDataProviders).
   *
   * @return 		true if transformation is skipped
   * @see		FilteredDataProvider
   */
  public boolean getRaw() {
    return m_Raw;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String rawTipText() {
    return
        "If set to true, then the raw data is returned instead of being "
      + "filtered through the global data container filter.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.Integer.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Integer.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the data to retrieve from the database
   */
  public abstract Class[] generates();

  /**
   * Returns the data provider to use for storing the container in the database.
   *
   * @return		the data provider
   */
  protected abstract DataProvider<T> getDataProvider();

  /**
   * Determines the database connection in the flow.
   *
   * @return		the database connection to use
   */
  protected abstract adams.db.AbstractDatabaseConnection getDatabaseConnection();

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null)
      m_DatabaseConnection = getDatabaseConnection();

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    T			cont;
    Integer		id;
    DataProvider<T>	provider;

    result = null;

    provider = getDataProvider();
    id       = (Integer) m_InputToken.getPayload();
    if (m_Raw)
      cont = ((FilteredDataProvider<T>) provider).loadRaw(id);
    else
      cont = provider.load(id);
    if (cont == null)
      result = "No container loaded for ID: " + m_InputToken;
    else
      m_OutputToken = new Token(cont);

    if (m_OutputToken != null)
      updateProvenance(m_OutputToken);

    return result;
  }

  /**
   * Updates the provenance information in the provided container.
   *
   * @param cont	the provenance container to update
   */
  public void updateProvenance(ProvenanceContainer cont) {
    if (Provenance.getSingleton().isEnabled())
      cont.addProvenance(new ProvenanceInformation(ActorType.DATAGENERATOR, this, m_OutputToken.getPayload().getClass()));
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    m_DatabaseConnection = null;

    super.wrapUp();
  }
}
