/*
 * AbstractSetReportValue.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import adams.data.report.AbstractField;
import adams.data.report.Field;
import adams.data.report.Report;
import adams.data.report.ReportHandler;
import adams.flow.core.Token;

/**
 * Ancestor for transformers that update the value of field in a report.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3528 $
 * @param <T> the type of field to handle
 */
public abstract class AbstractSetReportValue
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -5937471470417243026L;

  /** the field to get from the report. */
  protected AbstractField m_Field;

  /** the value to set. */
  protected String m_Value;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public abstract String globalInfo();

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "field", "field",
	    getDefaultField());

    m_OptionManager.add(
	    "value", "value",
	    "");
  }

  /**
   * Returns the default field for the option.
   *
   * @return		the default field
   */
  protected abstract AbstractField getDefaultField();

  /**
   * Sets the value to set in the report.
   *
   * @param value	the value to set
   */
  public void setValue(String value) {
    m_Value = value;
    reset();
  }

  /**
   * Returns the value to set in the report.
   *
   * @return		the value to set
   */
  public String getValue() {
    return m_Value;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public abstract String valueTipText();

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("field");
    if (variable != null)
      result = variable;
    else
      result = m_Field.toParseableString();

    result += " -> ";

    variable = getOptionManager().getVariableForProperty("value");
    if (variable != null)
      result += variable;
    else
      result += m_Value;

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the accepted classes
   */
  public abstract Class[] accepts();

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the generated classes
   */
  public abstract Class[] generates();

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Report	report;
    Field	field;

    result = null;

    if (m_InputToken.getPayload() instanceof ReportHandler)
      report = ((ReportHandler) m_InputToken.getPayload()).getReport();
    else
      report = (Report) m_InputToken.getPayload();

    try {
      if (report != null) {
	field = new Field(m_Field);
	report.addField(field);
	report.addParameter(m_Field.getName(), m_Value);
      }
      else {
	if (isDebugOn())
	  debug("No report available: " + m_InputToken);
      }
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    // broadcast data
    m_OutputToken = new Token(m_InputToken.getPayload());

    return result;
  }
}
