/*
 * ReportFileChooser.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.chooser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import adams.core.io.PlaceholderFile;
import adams.data.io.input.AbstractReportReader;
import adams.data.io.output.AbstractReportWriter;
import adams.data.report.Report;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.goe.GenericObjectEditor;
import adams.gui.goe.GenericObjectEditorDialog;

/**
 * A specialized JFileChooser that lists all available file Readers and Writers
 * for reports.
 * <p/>
 * Based on <code>weka.gui.ConverterFileChooser</code>
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3976 $
 * @see	    weka.gui.ConverterFileChooser
 * @param <T> the type of report
 */
public abstract class AbstractReportFileChooser<T extends Report>
  extends AbstractExtensionFileFilterFileChooser<AbstractReportFileChooser.ReportFileExtensionFilter> {

  /** for serialization. */
  private static final long serialVersionUID = -5373058011025481738L;

  /**
   * A custom filter class that stores the associated class along the
   * description and extensions.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 3976 $
   */
  public static class ReportFileExtensionFilter
    extends ExtensionFileFilter {

    /** for serialization. */
    private static final long serialVersionUID = -5589412775033677018L;

    /** the classname. */
    protected String m_Classname;

    /**
     * Constructs a filter that matches all files.
     *
     * @param classname		the classname this filter is for
     */
    public ReportFileExtensionFilter(String classname) {
      super();

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     */
    public ReportFileExtensionFilter(String classname, String description, String extension) {
      super(description, extension);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     */
    public ReportFileExtensionFilter(String classname, String description, String[] extensions) {
      super(description, extensions);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public ReportFileExtensionFilter(String classname, String description, String extension, boolean caseSensitive) {
      super(description, extension, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public ReportFileExtensionFilter(String classname, String description, String[] extensions, boolean caseSensitive) {
      super(description, extensions, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Returns the associated classname.
     *
     * @return		the classname
     */
    public String getClassname() {
      return m_Classname;
    }
  }

  /** the file filters for the readers. */
  protected static Hashtable<Class,Vector<ReportFileExtensionFilter>> m_ReaderFileFilters = new Hashtable<Class,Vector<ReportFileExtensionFilter>>();

  /** the file filters for the writers. */
  protected static Hashtable<Class,Vector<ReportFileExtensionFilter>> m_WriterFileFilters = new Hashtable<Class,Vector<ReportFileExtensionFilter>>();

  /** the configure button. */
  protected JButton m_ConfigureButton;

  /** the checkbox for bringing up the GenericObjectEditor. */
  protected JCheckBox m_CheckBoxOptions;

  /** the note about the options dialog. */
  protected JLabel m_LabelOptions;

  /** the GOE for displaying the options of a reader/writer. */
  protected GenericObjectEditor m_Editor;

  /**
   * onstructs a FileChooser pointing to the user's default directory.
   */
  protected AbstractReportFileChooser() {
    super();
  }

  /**
   * Constructs a FileChooser using the given File as the path.
   *
   * @param currentDirectory	the path to start in
   */
  protected AbstractReportFileChooser(File currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Constructs a FileChooser using the given path.
   *
   * @param currentDirectory	the path to start in
   */
  protected AbstractReportFileChooser(String currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Further initializations.
   */
  protected void initialize() {
    JPanel	panel;
    JPanel	panel2;

    super.initialize();

    m_CheckBoxOptions = new JCheckBox("Invoke options dialog");
    m_CheckBoxOptions.setMnemonic('I');
    m_LabelOptions = new JLabel("<html><br>Note:<br><br>Some file formats offer additional<br>options which can be customized<br>when invoking the options dialog.</html>");
    panel = new JPanel(new BorderLayout());
    panel.add(m_CheckBoxOptions, BorderLayout.NORTH);
    panel2 = new JPanel(new BorderLayout());
    panel2.add(m_LabelOptions, BorderLayout.NORTH);
    panel2.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panel.add(panel2, BorderLayout.CENTER);
    setAccessory(panel);

    m_Editor = new GenericObjectEditor(false);
  }

  /**
   * initializes the ReportFileExtensionFilters.
   *
   * @param chooser	the chooser instance to use as reference
   * @param reader	if true then the reader filters are initialized
   * @param classnames	the classnames of the converters
   */
  protected static void initFilters(AbstractReportFileChooser chooser, boolean reader, String[] classnames) {
    int				i;
    String 			classname;
    Class 			cls;
    String[] 			ext;
    String 			desc;
    Object		 	converter;
    ReportFileExtensionFilter 	filter;

    if (reader)
      m_ReaderFileFilters.put(chooser.getClass(), new Vector<ReportFileExtensionFilter>());
    else
      m_WriterFileFilters.put(chooser.getClass(), new Vector<ReportFileExtensionFilter>());

    for (i = 0; i < classnames.length; i++) {
      classname = (String) classnames[i];

      // get data from converter
      try {
	cls       = Class.forName(classname);
	converter = cls.newInstance();
	if (reader) {
	  desc = ((AbstractReportReader) converter).getFormatDescription();
	  ext  = ((AbstractReportReader) converter).getFormatExtensions();
	}
	else {
	  desc = ((AbstractReportWriter) converter).getFormatDescription();
	  ext  = ((AbstractReportWriter) converter).getFormatExtensions();
	}
      }
      catch (Exception e) {
	cls       = null;
	converter = null;
	ext       = new String[0];
	desc      = "";
      }

      if (converter == null)
	continue;

      // reader?
      if (reader) {
	filter = new ReportFileExtensionFilter(classname, desc, ext);
	m_ReaderFileFilters.get(chooser.getClass()).add(filter);
      }
      else {
	filter = new ReportFileExtensionFilter(classname, desc, ext);
	m_WriterFileFilters.get(chooser.getClass()).add(filter);
      }
    }
  }

  /**
   * Returns the file filters for opening files.
   *
   * @return		the file filters
   */
  protected Vector<ReportFileExtensionFilter> getOpenFileFilters() {
    return m_ReaderFileFilters.get(getClass());
  }

  /**
   * Returns the file filters for writing files.
   *
   * @return		the file filters
   */
  protected Vector<ReportFileExtensionFilter> getSaveFileFilters() {
    return m_WriterFileFilters.get(getClass());
  }

  /**
   * initializes the GUI.
   *
   * @param dialogType		the type of dialog to setup the GUI for
   */
  protected void initGUI(int dialogType) {
    super.initGUI(dialogType);

    // initial setup
    if (dialogType == OPEN_DIALOG) {
      m_Editor.setClassType(AbstractReportReader.class);
      m_Editor.setValue(getDefaultReader());
    }
    else {
      m_Editor.setClassType(AbstractReportWriter.class);
      m_Editor.setValue(getDefaultWriter());
    }
  }

  /**
   * Returns the default reader.
   *
   * @return		the default reader
   */
  protected abstract AbstractReportReader<T> getDefaultReader();

  /**
   * Returns the default writer.
   *
   * @return		the default writer
   */
  protected abstract AbstractReportWriter<T> getDefaultWriter();

  /**
   * Pops up an "Open File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  public int showOpenDialog(Component parent) {
    int result = super.showOpenDialog(parent);

    if (result == APPROVE_OPTION) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResult();
	if (result == GenericObjectEditorDialog.APPROVE_OPTION)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * Pops up an "Save File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  public int showSaveDialog(Component parent) {
    int result = super.showSaveDialog(parent);

    if (result == APPROVE_OPTION) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResult();
	if (result == GenericObjectEditorDialog.APPROVE_OPTION)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * returns the reader that was chosen by the user, can be null in case the
   * user aborted the dialog or the save dialog was shown.
   *
   * @return		the chosen reader, if any
   */
  public AbstractReportReader getReader() {
    configureCurrentHandlerHook(OPEN_DIALOG);

    if (m_CurrentHandler instanceof AbstractReportWriter)
      return null;
    else
      return (AbstractReportReader) m_CurrentHandler;
  }

  /**
   * returns the writer that was chosen by the user, can be null in case the
   * user aborted the dialog or the open dialog was shown.
   *
   * @return		the chosen writer, if any
   */
  public AbstractReportWriter getWriter() {
    configureCurrentHandlerHook(SAVE_DIALOG);

    if (m_CurrentHandler instanceof AbstractReportReader)
      return null;
    else
      return (AbstractReportWriter) m_CurrentHandler;
  }

  /**
   * sets the current converter according to the current filefilter.
   */
  protected void updateCurrentHandlerHook() {
    String	classname;
    Object	newHandler;

    try {
      // determine current converter
      classname  = ((ReportFileExtensionFilter) getFileFilter()).getClassname();
      newHandler = Class.forName(classname).newInstance();

      if (m_CurrentHandler == null) {
	m_CurrentHandler = newHandler;
      }
      else {
	if (!m_CurrentHandler.getClass().equals(newHandler.getClass()))
	  m_CurrentHandler = newHandler;
      }

      setFileSelectionMode(FILES_ONLY);
    }
    catch (Exception e) {
      m_CurrentHandler = null;
      e.printStackTrace();
    }
  }

  /**
   * configures the current converter.
   *
   * @param dialogType		the type of dialog to configure for
   */
  protected void configureCurrentHandlerHook(int dialogType) {
    PlaceholderFile	selFile;
    String		classname;
    File		currFile;

    selFile = getSelectedPlaceholderFile();

    if (m_CurrentHandler == null) {
      classname = ((ReportFileExtensionFilter) getFileFilter()).getClassname();
      try {
	m_CurrentHandler = Class.forName(classname).newInstance();
      }
      catch (Exception e) {
	e.printStackTrace();
	m_CurrentHandler = null;
      }

      // none found?
      if (m_CurrentHandler == null)
	return;
    }

    // wrong type?
    if (selFile.isDirectory())
      return;

    try {
      if (m_CurrentHandler instanceof AbstractReportReader)
	currFile = ((AbstractReportReader) m_CurrentHandler).getInput();
      else
	currFile = ((AbstractReportWriter) m_CurrentHandler).getOutput();
      if ((currFile == null) || (!currFile.getAbsolutePath().equals(selFile.getAbsolutePath()))) {
	if (m_CurrentHandler instanceof AbstractReportReader)
	  ((AbstractReportReader) m_CurrentHandler).setInput(selFile);
	else
	  ((AbstractReportWriter) m_CurrentHandler).setOutput(selFile);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Returns whether the filters have already been initialized.
   *
   * @return		true if the filters have been initialized
   */
  protected boolean getFiltersInitialized() {
    return (m_ReaderFileFilters.containsKey(getClass()));
  }
}
