/*
 * BaseWindow.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.core;

import javax.swing.JWindow;

import adams.core.PrintObject;
import adams.core.PrintObject.OutputType;
import adams.core.option.OptionUtils;
import adams.env.Environment;

/**
 * A window that loads the size and location from the props file automatically.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4397 $
 */
public class BaseWindow
  extends JWindow {

  /** for serialization. */
  private static final long serialVersionUID = 6176039693692479862L;

  /** the object for printing to stdout. */
  protected PrintObject m_SystemOut;

  /** the object for printing to stderr. */
  protected PrintObject m_SystemErr;

  /** the object for debugging output. */
  protected PrintObject m_Debugging;

  /**
   * Initializes the frame with no title.
   */
  public BaseWindow() {
    super();
    performInitialization();
  }

  /**
   * Contains all the initialization steps to perform.
   */
  protected void performInitialization() {
    initialize();
    initGUI();
    finishInit();
  }

  /**
   * For initializing members.
   */
  protected void initialize() {
    m_SystemOut = new PrintObject(this, OutputType.STDOUT, true, null);
    m_SystemErr = new PrintObject(this, OutputType.STDERR, true, null);
    m_Debugging = new PrintObject(this, OutputType.DEBUG, false, null);
  }

  /**
   * For initializing the GUI.
   */
  protected void initGUI() {
    if (GUIHelper.getLogoIcon() != null)
      setIconImage(GUIHelper.getLogoIcon().getImage());
  }

  /**
   * finishes the initialization, by setting size/location.
   */
  protected void finishInit() {
    // size and location
    GUIHelper.setSizeAndLocation(this, this);
  }

  /**
   * Hook method just before the dialog is made visible.
   */
  protected void beforeShow() {
  }

  /**
   * Hook method just after the dialog was made visible.
   */
  protected void afterShow() {
  }

  /**
   * Hook method just before the dialog is hidden.
   */
  protected void beforeHide() {
  }

  /**
   * Hook method just after the dialog was hidden.
   */
  protected void afterHide() {
  }

  /**
   * closes/shows the dialog.
   *
   * @param value	if true then display the dialog, otherwise close it
   */
  public void setVisible(boolean value) {
    if (value)
      beforeShow();
    else
      beforeHide();

    super.setVisible(value);

    if (value)
      afterShow();
    else
      afterHide();
  }

  /**
   * Instantiates the frame.
   *
   * @param classname	the classname of the frame to instantiate
   * @return		the instantiated frame or null if an error occurred
   */
  public static BaseWindow forName(String classname) {
    BaseWindow	result;

    try {
      result = (BaseWindow) OptionUtils.forName(AbstractFrameWithOptionHandling.class, classname, new String[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Runs the window from the commandline.
   *
   * @param env		the environment class to use
   * @param app		the window class
   */
  public static void runWindow(Class env, Class app) {
    BaseWindow	frameInst;

    Environment.setEnvironmentClass(env);

    frameInst = forName(app.getName());
    frameInst.setVisible(true);
  }
}
