/**
 * TextFileWriter.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;


/**
 <!-- globalinfo-start -->
 * Writes the content to a text file.Makes sure that the filename is valid.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-dir &lt;adams.core.io.PlaceholderFile&gt; (property: directory)
 * &nbsp;&nbsp;&nbsp;The directory to save the outputs in.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-prefix &lt;java.lang.String&gt; (property: prefix)
 * &nbsp;&nbsp;&nbsp;The prefix for the output files.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-suffix &lt;java.lang.String&gt; (property: suffix)
 * &nbsp;&nbsp;&nbsp;The suffix for the output files.
 * &nbsp;&nbsp;&nbsp;default: .txt
 * </pre>
 *
 * <pre>-ignore-name (property: ignoreName)
 * &nbsp;&nbsp;&nbsp;If set to true, then the name of the content is ignored for generating the
 * &nbsp;&nbsp;&nbsp;filename (useful when prefix or suffix is based on variables).
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2391 $
 */
public class TextFileWriter
  extends AbstractTextWriter {

  /** for serialization. */
  private static final long serialVersionUID = 2481561390856324348L;

  /** the output directory. */
  protected PlaceholderFile m_Directory;

  /** the output prefix. */
  protected String m_Prefix;

  /** the output suffix. */
  protected String m_Suffix;

  /** whether to discard the provided name and rely solely on prefix/suffix
   * (useful if prefix/suffix are based on variables). */
  protected boolean m_IgnoreName;

  /**
   * Returns a short description of the writer.
   *
   * @return		a description of the writer
   */
  public String globalInfo() {
    return
        "Writes the content to a text file."
      + "Makes sure that the filename is valid.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "dir", "directory",
	    new PlaceholderFile("."));

    m_OptionManager.add(
	    "prefix", "prefix",
	    "");

    m_OptionManager.add(
	    "suffix", "suffix",
	    ".txt");

    m_OptionManager.add(
	    "ignore-name", "ignoreName",
	    false);
  }

  /**
   * Sets the output directory.
   *
   * @param value 	the directory for the outputs
   */
  public void setDirectory(PlaceholderFile value) {
    m_Directory = value;
    reset();
  }

  /**
   * Returns the output directory.
   *
   * @return 		the directory for the outputs
   */
  public PlaceholderFile getDirectory() {
    return m_Directory;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String directoryTipText() {
    return "The directory to save the outputs in.";
  }

  /**
   * Sets the prefix for the output files.
   *
   * @param value 	the prefix
   */
  public void setPrefix(String value) {
    m_Prefix = value;
    reset();
  }

  /**
   * Returns the prefix for the output files.
   *
   * @return 		the prefix
   */
  public String getPrefix() {
    return m_Prefix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String prefixTipText() {
    return "The prefix for the output files.";
  }

  /**
   * Sets the suffix for the output files.
   *
   * @param value 	the suffix
   */
  public void setSuffix(String value) {
    m_Suffix = value;
    reset();
  }

  /**
   * Returns the suffix for the output files.
   *
   * @return 		the suffix
   */
  public String getSuffix() {
    return m_Suffix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String suffixTipText() {
    return "The suffix for the output files.";
  }

  /**
   * Sets whether to ignore the name and just rely on prefix/suffix to
   * generate the filename.
   *
   * @param value 	if true then the name of the content gets ignored
   */
  public void setIgnoreName(boolean value) {
    m_IgnoreName = value;
    reset();
  }

  /**
   * Returns whether the name of the content is ignored and the filename is
   * only generated based on prefix/suffix.
   *
   * @return 		true if the name of the content gets ignored
   */
  public boolean getIgnoreName() {
    return m_IgnoreName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String ignoreNameTipText() {
    return
        "If set to true, then the name of the content is ignored for "
      + "generating the filename (useful when prefix or suffix is based on "
      + "variables).";
  }

  /**
   * Creates the filename.
   *
   * @param name	the name of the content
   * @return		the generated filename
   */
  protected String createFilename(String name) {
    String	result;

    if (m_IgnoreName)
      result =   getDirectory().getAbsolutePath()
               + PlaceholderFile.separator
               + getPrefix()
               + getSuffix();
    else
      result =   getDirectory().getAbsolutePath()
               + PlaceholderFile.separator
               + getPrefix()
               + FileUtils.createFilename(name, "_")
               + getSuffix();

    return result;
  }

  /**
   * Writes the given content under the specified name.
   *
   * @param content	the content to write
   * @param name	the name under which to save the content
   * @return		if a file was generated, the filename the content was written
   * 			as, otherwise null
   */
  public String write(String content, String name) {
    boolean	ok;
    String	filename;

    filename = createFilename(name);
    if (isDebugOn())
      debug("Filename: " + filename);

    ok = FileUtils.writeToFile(filename, content, false);
    if (isDebugOn())
      debug("Result: " + ok);

    if (ok)
      return filename;
    else
      return null;
  }
}
