/*
 * SQL.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.db;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import adams.core.ConsoleObject;
import adams.core.DateFormat;
import adams.core.DateUtils;
import adams.core.Utils;
import adams.core.base.BaseRegExp;

/**
 * Basic SQL support.
 *
 * @author dale
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4465 $
 */
public class SQL
  extends ConsoleObject
  implements DatabaseConnectionProvider {

  /** for serialization. */
  private static final long serialVersionUID = -7708896486343190549L;

  /** whether debugging is turned on. */
  protected boolean m_Debug;

  /** connection to database. */
  protected AbstractDatabaseConnection m_DatabaseConnection;

  /** the table manager. */
  protected static TableManager<SQL> m_TableManager;

  /**
   * Constructor.
   *
   * @param dbcon	the database context to use
   */
  public SQL(AbstractDatabaseConnection dbcon) {
    super();

    m_DatabaseConnection = dbcon;

    updatePrefix();
  }

  /**
   * Updates the prefix of the console object output streams.
   */
  protected void updatePrefix() {
    String	prefix;

    prefix = getClass().getName() + "(" + getDatabaseConnection().toStringShort() + "/" + getDatabaseConnection().hashCode() + ")";

    getSystemOut().setPrefix(prefix);
    getSystemErr().setPrefix(prefix);
    getDebugging().setPrefix(prefix);
  }

  /**
   * Returns the database connection this table is for.
   *
   * @return		the database connection
   */
  public AbstractDatabaseConnection getDatabaseConnection() {
    return m_DatabaseConnection;
  }

  /**
   * Sets whether debugging is enabled, outputs more on the console.
   *
   * @param value	if true debugging is enabled
   */
  public void setDebug(boolean value) {
    m_Debug = value;
    getDebugging().setEnabled(value);
  }

  /**
   * Returns whether debugging is enabled.
   *
   * @return		true if debugging is enabled
   */
  public boolean getDebug() {
    return m_Debug;
  }

  /**
   * Processes the debugging message.
   *
   * @param msg		the debugging message to process
   */
  protected void debug(String msg) {
    getDebugging().println(msg);
  }

  /**
   * Replaces all single quotes -' with double quotes -".
   *
   * @param in  Input String to replace quotes
   * @return  String with quotes replaced
   */
  public String escapeQuotes(String in) {
    String ret=in.replaceAll("'","\"");
    return(ret);
  }

  /**
   * Checks that a given table exists.
   *
   * @param table	the table to look for
   * @return true if the table exists.
   */
  public boolean tableExists(String table) {
    boolean tableExists = false;
    ResultSet rs = null;
    Connection connection = m_DatabaseConnection.getConnection(true);
    try{
      DatabaseMetaData dbmd = connection.getMetaData();
      rs = dbmd.getTables (null, null, table, null);
      tableExists = rs.next();
    }
    catch (SQLException e) {
      // try again
      try {
	DatabaseMetaData dbmd = connection.getMetaData();
	rs = dbmd.getTables (null, null, table, null);
	tableExists = rs.next();
      }
      catch (Exception ex) {
	// ignored?
      }
    }
    catch (Exception e) {
      // ignored?
    }
    finally{
      if (rs!=null) {
	try{
	  rs.close();
	}
	catch (Exception e) {
	  // ignored?
	}
      }
    }
    return tableExists;
  }

  /**
   * Execute the given SQL statement and return ResultSet.
   *
   * @param  query  SQL query String
   * @return resulting ResultSet, or null if Error
   * @throws Exception if something goes wrong
   */
  public SimpleResultSet getSimpleResultSet(String query) throws Exception {
    return(new SimpleResultSet(getResultSet(query)));
  }

  /**
   * Create a Prepared statement with given query.
   *
   * @param query 	the query to execute
   * @return 		PreparedStatement
   * @throws Exception 	if something goes wrong
   */
  public PreparedStatement prepareStatement(String query) throws Exception{
    return prepareStatement(query, false);
  }

  /**
   * Create a Prepared statement with given query.
   *
   * @param query 	the query to execute
   * @param returnKeys 	whether to initialize the statement that it returns
			the generated keys
   * @return 		PreparedStatement
   * @throws Exception 	if something goes wrong
   */
  public PreparedStatement prepareStatement(String query, boolean returnKeys) throws Exception{
    Connection connection = m_DatabaseConnection.getConnection(true);
    PreparedStatement stmt = null;
    debug("Preparing statement for: " + query);
    try {
      if (returnKeys)
	stmt = connection.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
      else
	stmt = connection.prepareStatement(query);
    }
    catch (SQLException e) {
      // try again
      if (returnKeys)
	stmt = connection.prepareStatement(query, Statement.RETURN_GENERATED_KEYS);
      else
	stmt = connection.prepareStatement(query);
    }
    catch (Exception e) {
      getSystemErr().println("Error preparing statement for: " + query);
      getSystemErr().printStackTrace(e);
      throw new Exception(e);
    }
    return(stmt);
  }

  /**
   * Update table.
   *
   * @param updateString 	comma separated updates. e.g weight='80',height=180
   * @param table		the table to update
   * @param where		condition. e.g name='bob smith'
   * @return			number of rows affected
   * @throws Exception 		if something goes wrong
   */
  public int update(String updateString, String table, String where) throws Exception{
    String query="UPDATE " + table + " SET " + updateString + " WHERE " + where;
    Connection connection = m_DatabaseConnection.getConnection(true);
    Statement stmt = null;
    debug("Updating: " + query);
    int uc = 0;
    try {
      stmt = connection.createStatement();
      stmt.execute(query);
    }
    catch (SQLException e) {
      // try again
      try {
	if (stmt != null)
	  stmt.close();
	stmt = connection.createStatement();
	stmt.execute(query);
      }
      catch (Exception ex) {
	getSystemErr().println("Error executing 'update': " + query);
	getSystemErr().printStackTrace(ex);
	return(-1);
      }
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      return(-1);
    }
    finally {
      if (stmt != null) {
	uc = stmt.getUpdateCount();
	stmt.close();
      }
    }
    return(uc);
  }

  /**
   * Executes a SQL query. Return any Generated Keys
   * Caller is responsible for closing the statement.
   *
   * @param query the SQL query
   * @return Generated keys as a resultset, or null if failure
   * @throws Exception if something goes wrong
   */
  public ResultSet executeGeneratedKeys(String query) throws Exception {
    Connection connection = m_DatabaseConnection.getConnection(true);
    Statement stmt = null;
    debug("Execute generated keys: " + query);
    try {
      stmt = connection.createStatement();
      stmt.execute(query, Statement.RETURN_GENERATED_KEYS);
      return(stmt.getGeneratedKeys());
    }
    catch (SQLException e) {
      getSystemErr().println("Error executing 'executeGeneratedKeys': " + query);
      getSystemErr().printStackTrace(e);
      if (stmt != null)
	stmt.close();
      return(null);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      if (stmt != null)
	stmt.close();
      return(null);
    }
  }

  /**
   * Executes a SQL query.
   *
   * @param query the SQL query
   * @return true if the query generated results
   * @throws Exception if an error occurs
   */
  public boolean execute(String query) throws Exception {
    Connection 	connection;
    Statement 	stmt;
    boolean 	result;

    connection = m_DatabaseConnection.getConnection(true);
    if (connection == null)
      throw new IllegalStateException(
	  "Connection object is null (" + m_DatabaseConnection.toStringShort() + "/" + m_DatabaseConnection.hashCode() + ")!");
    stmt = null;
    debug("Execute: " + query);
    try {
      stmt   = connection.createStatement();
      result = stmt.execute(query);
    }
    catch (SQLException e) {
      // try again
      try {
	if (stmt != null)
	  stmt.close();
	stmt   = connection.createStatement();
	result = stmt.execute(query);
      }
      catch (Exception ex) {
	getSystemErr().println("Error executing 'execute': " + query);
	getSystemErr().printStackTrace(ex);
	result = false;
      }
    }
    catch (Exception e) {
      getSystemErr().println("Error executing query: " + query);
      getSystemErr().printStackTrace(e);
      result = false;
    }
    finally {
      close(stmt);
    }

    return result;
  }

  /**
   * Empty table.
   *
   * @param table	the table to empty
   * @return		success?
   */
  public boolean truncate(String table) {
    boolean	result;

    try{
      execute("TRUNCATE TABLE " + table);
      result = true;
    }
    catch(Exception e) {
      getSystemErr().println("Error truncating table '" + table + "':");
      getSystemErr().printStackTrace(e);
      result = false;
    }

    return result;
  }

  /**
   * Drops the table.
   *
   * @param table	the table to empty
   * @return		success?
   */
  public boolean drop(String table) {
    boolean	result;

    try{
      execute("DROP TABLE " + table);
      result = true;
    }
    catch(Exception e) {
      getSystemErr().println("Error dropping table '" + table + "':");
      getSystemErr().printStackTrace(e);
      result = false;
    }

    return result;
  }

  /**
   * Close this statement to avoid memory leaks.
   *
   * @param s		the statement to close
   */
  public static void close(Statement s) {
    if (s != null) {
      try {
	s.close();
	s = null;
      }
      catch (Exception e) {
	System.err.println("Error closing statement: " + e.toString());
      }
    }
  }

  /**
   * Close objects related to this ResultSet. Important because some (most,all?) jdbc drivers
   * do not clean up after themselves, resulting in memory leaks.
   *
   * @param r  The ResultSet to clean up after
   */
  public static void closeAll(ResultSet r) {
    if (r != null) {
      try {
	Statement s = r.getStatement();
	r.close();
	close(s);
	s = null;
	r = null;
      }
      catch (Exception e) {
	System.err.println("Error closing resultset: " + e.toString());
      }
    }
  }

  /**
   * Close objects related to this ResultSet.
   *
   * @param r  The ResultSet to clean up after
   */
  public static void closeAll(SimpleResultSet r) {
    if (r != null) {
      try {
	r.close();
      }
      catch (Exception e) {
	System.err.println("Error closing resultset/statement: " + e.toString());
      }
    }
  }

  /**
   * Do a select on given columns for all data in joined tables, with condition.
   *
   * @param columns	columns to select
   * @param tables	the tables to select from
   * @param where	condition
   * @return		resultset of data
   * @throws Exception 	if something goes wrong
   */
  public ResultSet select(String columns, String tables, String where) throws Exception {
    return doSelect(false, columns, tables, where);
  }

  /**
   * Do a select distinct on given columns for all data in joined tables, with
   * condition.
   *
   * @param columns	columns to select
   * @param tables	the tables to select from
   * @param where	condition
   * @return		resultset of data
   * @throws Exception 	if something goes wrong
   */
  public ResultSet selectDistinct(String columns, String tables, String where) throws Exception{
    return doSelect(true, columns, tables, where);
  }

  /**
   * Do a select on given columns for all data in joined tables, with condition.
   * Can be distinct.
   *
   * @param distinct	whether values in columns has to be distinct
   * @param columns	columns to select
   * @param tables	the tables to select from
   * @param where	condition
   * @return		resultset of data
   * @throws Exception 	if something goes wrong
   */
  protected ResultSet doSelect(boolean distinct, String columns, String tables, String where) throws Exception {
    String	query;

    // select
    query = "SELECT ";
    if (distinct)
      query += "DISTINCT ";
    query += columns;

    // from
    query += " FROM " + tables;

    // where
    if ((where != null) && (where.length() > 0)) {
      if (   !where.trim().toUpperCase().startsWith("LIMIT ")
	  && !where.trim().toUpperCase().startsWith("ORDER ") )
	query += " WHERE";
      query += " " + where;
    }

    debug("doSelect: " + query);
    try {
      return getResultSet(query);
    }
    catch (SQLException e) {
      getSystemErr().println("Error executing 'doSelect': " + query);
      getSystemErr().printStackTrace(e);
      throw e;
    }
  }

  /**
   * MySQL boolean to tinyint.
   *
   * @param b	boolean
   * @return	tiny int value
   */
  public static int booleanToTinyInt(boolean b) {
    if (b) {
      return(1);
    } else {
      return(0);
    }
  }

  /**
   * MySQL tinyint to boolean.
   * @param i	tiny int
   * @return	boolean
   */
  public static boolean tinyIntToBoolean(int i) {
    if (i==0) {
      return(false);
    } else {
      return(true);
    }
  }

  /**
   * Backquotes the regular expression and ensures that it is surrounded by single
   * quotes.
   *
   * @param s		the regular expression to backquote and enclose
   * @return		the processed string
   */
  public static String backquote(BaseRegExp s) {
    return backquote(s.getValue());
  }

  /**
   * Backquotes the string and ensures that it is surrounded by single
   * quotes.
   *
   * @param s		the string to backquote and enclose
   * @return		the processed string
   */
  public static String backquote(String s) {
    String	result;

    result = Utils.backQuoteChars(s);
    if (!result.startsWith("'"))
      result = "'" + result + "'";

    return result;
  }

  /**
   * Return resultset of given query.
   *
   * @param query	sql query
   * @return resulset
   * @throws Exception if something goes wrong
   */
  public ResultSet getResultSet(String query) throws Exception {
    Connection connection = m_DatabaseConnection.getConnection(true);
    debug("Get ResultSet for : " + query);
    if (connection == null)
      throw new IllegalStateException("Connection object is null!");

    Statement stmt = null;
    try {
      stmt = connection.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,ResultSet.CONCUR_READ_ONLY);
    }
    catch (SQLException e) {
      // try again
      stmt = connection.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,ResultSet.CONCUR_READ_ONLY);
    }
    return(stmt.executeQuery(query));
  }

  /**
   * Returns a short string representation.
   *
   * @return		the string representation
   */
  public String toString() {
    return "SQL: " + getDatabaseConnection().toString();
  }

  /**
   * Returns a new SimpleDateFormat that can be used for timestamps.
   *
   * @return		the initialied date format
   */
  public static DateFormat getTimestampDateFormat() {
    return DateUtils.getTimestampFormatter();
  }

  /**
   * Returns the singleton instance.
   *
   * @return		the singleton
   */
  public synchronized static SQL getSingleton(AbstractDatabaseConnection dbcon) {
    if (m_TableManager == null)
      m_TableManager = new TableManager<SQL>("SQL", null);
    if (!m_TableManager.has(dbcon))
      m_TableManager.add(dbcon, new SQL(dbcon));

    return m_TableManager.get(dbcon);
  }
}
