/*
 * SequentialDirector.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;


import java.util.Stack;
import java.util.Vector;

import adams.flow.core.AbstractActor;
import adams.flow.core.ActorUtils;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;
import adams.flow.core.Token;

/**
 * Manages the execution of actors in sequential order.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3154 $
 */
public class SequentialDirector
  extends AbstractDirector {

  /** for serialization. */
  private static final long serialVersionUID = -1634725837304059804L;

  /** for storing the token that the last actor generated. */
  protected transient Vector<Token> m_FinalOutput;

  /** whether execution has finished. */
  protected boolean m_Finished;

  /** whether the director was executed at all. */
  protected boolean m_Executed;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_FinalOutput = null;
  }

  /**
   * Returns whether the final output of actors is recorded.
   * <p/>
   * Default implementation returns false.
   *
   * @return		true if final output is to be recorded
   */
  protected boolean isFinalOutputRecorded() {
    return false;
  }

  /**
   * Returns the vector for storing final outputs. Gets instantiated if
   * necessary.
   *
   * @return		the vector for storing the tokens
   * @see		#m_FinalOutput
   */
  protected Vector<Token> getFinalOutput() {
    if (m_FinalOutput == null)
      m_FinalOutput = new Vector<Token>();

    return m_FinalOutput;
  }

  /**
   * Checks whether the actor has already stopped. If so, outputs a message
   * on the commandline and returns the error message.
   *
   * @param actor	the actor to check
   * @return		null if actor not stopped, otherwise the error message
   */
  protected String checkActorHasStopped(AbstractActor actor) {
    String	result;
    Throwable	th;

    result = null;

    if (actor.isStopped()) {
      th = new Throwable();
      th.fillInStackTrace();
      result = th.getStackTrace()[1].getMethodName() + ": Actor '" + actor.getFullName() + "' is stopped!";
    }

    return result;
  }

  /**
   * Presents the specified token to the actor.
   *
   * @param actor	the actor to use (InputConsumer)
   * @param input	the input token
   * @return		the error message, null if everything OK
   */
  protected String doInput(AbstractActor actor, Token input) {
    String	result;
    String	msg;

    result = null;

    if ((msg = checkActorHasStopped(actor)) != null)
      return msg;

    try {
      ((InputConsumer) actor).input(input);
    }
    catch (Exception e) {
      msg = "Actor '" + actor.getFullName() + "' generated the following "
	  + "exception on calling the 'input(Token)' method with token '" + input + "': ";
      result = msg + e;
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
    }

    if (result != null)
      actor.handleError("input", result);

    return result;
  }

  /**
   * Calls the execute() method of the actor.
   *
   * @param actor	the actor to use
   * @return		a potential error message
   */
  protected String doExecute(AbstractActor actor) {
    String	result;
    String	msg;

    if ((msg = checkActorHasStopped(actor)) != null)
      return msg;

    try {
      if (getDebugLevel() >= 10)
	debug("Size before 'execute()': " + actor.sizeOf() + " [" + actor.getFullName() + "]", 10);
      result = actor.execute();
      if (getDebugLevel() >= 10)
	debug("Size after 'execute()': " + actor.sizeOf() + " [" + actor.getFullName() + "]", 10);
    }
    catch (Exception e) {
      msg = "Actor '" + actor.getFullName() + "' generated the following "
	  + "exception on calling the 'execute()' method: ";
      result = msg + e;
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
    }

    if (result != null)
      actor.handleError("execute", result);

    return result;
  }

  /**
   * Checks whether the actor has pending output.
   *
   * @param actor	the actor to use (OutputProducer)
   * @return		the token that was retrieved from the actor, null
   * 			in case of an error and not 'stopping on errors'
   * @see		AbstractDirectedControlActor#getStopOnErrors()
   */
  protected boolean doHasOutput(AbstractActor actor) {
    boolean	result;
    String	msg;
    String	msgFull;

    if (checkActorHasStopped(actor) != null)
      return false;

    msgFull = null;

    try {
      result = ((OutputProducer) actor).hasPendingOutput();
    }
    catch (Exception e) {
      result = false;
      msg = "Actor '" + actor.getFullName() + "' generated the following "
	  + "exception on calling the 'hasPendingOutput()' method: ";
      msgFull = msg + e;
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
    }

    if (msgFull != null)
      actor.handleError("hasPendingOutput", msgFull);

    return result;
  }

  /**
   * Retrieves the token from the actor.
   *
   * @param actor	the actor to use
   * @return		the token that was retrieved from the actor, null
   * 			in case of an error and not 'stopping on errors'
   * @see		AbstractDirectedControlActor#getStopOnErrors()
   */
  protected Token doOutput(AbstractActor actor) {
    Token	result;
    String	msg;
    String	msgFull;

    if (checkActorHasStopped(actor) != null)
      return null;

    msgFull = null;

    try {
      if (getDebugLevel() >= 10)
	debug("Size before 'output()': " + actor.sizeOf() + " [" + actor.getFullName() + "]", 10);
      result = ((OutputProducer) actor).output();
      if (getDebugLevel() >= 10)
	debug("Size after 'output()': " + actor.sizeOf() + " [" + actor.getFullName() + "]", 10);
    }
    catch (Exception e) {
      result = null;
      msg = "Actor '" + actor.getFullName() + "' generated the following "
	  + "exception on calling the 'output()' method: ";
      msgFull = msg + e;
      getSystemErr().println(msg);
      getSystemErr().printStackTrace(e);
    }

    if (msgFull != null)
      actor.handleError("output", msgFull);

    return result;
  }

  /**
   * Executes all the standalone actors. Returns the first non-standalone actor.
   *
   * @return		the first non-standalone actor or null if non present
   */
  protected AbstractActor doExecuteStandalones() {
    AbstractActor	result;
    AbstractActor	curr;
    int			i;
    String		actorResult;

    result = null;

    for (i = 0; i < m_ControlActor.size(); i++) {
      // paused?
      if (m_Paused)
	pause();

      // stopped?
      if (isStopped() || isStopping())
	break;

      curr = m_ControlActor.get(i);
      if (curr.getSkip())
	continue;

      if (!ActorUtils.isStandalone(curr)) {
	result = curr;
	break;
      }
      else {
	actorResult = doExecute(curr);
	if (actorResult != null)
	  getSystemErr().println(
	      curr.getFullName() + " generated following error output:\n"
	      + actorResult);
      }
    }

    return result;
  }

  /**
   * Peforms the execution of the actors.
   *
   * @param startActor	the actor to start with
   * @return		null if everything ok, otherwise the error message
   */
  protected String doExecuteActors(AbstractActor startActor) {
    String			result;
    boolean			finished;
    int				startIndex;
    int				i;
    AbstractActor		notFinishedActor;
    Stack<AbstractActor>	pendingActors;
    Token			token;
    AbstractActor		curr;
    String			actorResult;
    int				lastActive;

    result           = null;
    notFinishedActor = startActor;
    pendingActors    = new Stack<AbstractActor>();
    getFinalOutput().clear();
    do {
      if (isDebugOn())
	debug("--> iteration start");

      // paused?
      if (isPaused())
	pause();

      // do we have to stop the execution?
      if (isStopped() || isStopping())
	break;

      // determing starting point of next iteration
      if (pendingActors.size() > 0) {
	startIndex = m_ControlActor.indexOf(pendingActors.peek().getName());
      }
      else {
	startIndex       = m_ControlActor.indexOf(notFinishedActor.getName());
	notFinishedActor = null;
      }
      if (isDebugOn())
	debug("Start index: " + startIndex, 2);

      // iterate over actors
      curr       = null;
      token      = null;
      lastActive = -1;
      if (m_ControlActor.active() > 0)
	lastActive = m_ControlActor.lastActive().index();
      for (i = startIndex; i <= lastActive; i++) {
	// paused?
	if (isPaused())
	  pause();

	// do we have to stop the execution?
	if (isStopped() || isStopping())
	  break;

	curr = m_ControlActor.get(i);
	if (curr.getSkip())
	  continue;
	if (isDebugOn())
	  debug("Current actor: " + curr.getFullName(), 2);

	// no token? get pending one or produce new one
	if (token == null) {
	  if ((curr instanceof OutputProducer) && doHasOutput(curr)) {
	    pendingActors.pop();
	    debug("Actor holds another output token: " + curr.getFullName(), 2);
	  }
	  else {
	    actorResult = doExecute(curr);
	    if (actorResult != null) {
	      getSystemErr().println(
		  curr.getFullName() + " generated following error output:\n"
		  + actorResult);
	      if (curr.getStopFlowOnError())
		break;
	    }
	    if (!curr.isFinished() && (notFinishedActor == null))
	      notFinishedActor = curr;
	    if (isDebugOn())
	      debug("Actor needed to be executed: " + curr.getFullName(), 2);
	  }

	  if ((curr instanceof OutputProducer) && doHasOutput(curr))
	    token = doOutput(curr);
	  else
	    token = null;
	  if (getDebugLevel() > 2)
	    debug("Token obtained from output: " + token, 3);
	  else
	    debug("Token obtained from output", 2);

	  // still more to come?
	  if ((curr instanceof OutputProducer) && doHasOutput(curr)) {
	    pendingActors.push(curr);
	    if (isDebugOn())
	      debug("Actor has more tokens on output: " + curr.getFullName(), 2);
	  }
	}
	else {
	  // process token
	  doInput(curr, token);
	  actorResult = doExecute(curr);
	  if (actorResult != null) {
	    getSystemErr().println(
		curr.getFullName() + " generated following error output:\n"
		+ actorResult);
	    if (curr.getStopFlowOnError())
	      break;
	  }
	  if (!curr.isFinished() && (notFinishedActor == null))
	    notFinishedActor = curr;
	  if (getDebugLevel() > 2)
	    debug("Actor processes token: " + curr.getFullName() + "/" + token, 3);
	  else
	    debug("Actor processes token: " + curr.getFullName(), 2);

	  // was a new token produced?
	  if (curr instanceof OutputProducer) {
	    if (doHasOutput(curr))
	      token = doOutput(curr);
	    else
	      token = null;
	    if (isDebugOn())
	      debug("Actor also produces tokens: " + curr.getFullName(), 2);

	    // still more to come?
	    if (doHasOutput(curr)) {
	      if (isDebugOn())
		debug("Actor also has more tokens on output: " + curr.getFullName(), 2);
	      pendingActors.push(curr);
	    }
	  }
	  else {
	    token = null;
	  }
	}

	// token from last actor generated? -> store
	if ((i == m_ControlActor.lastActive().index()) && (token != null)) {
	  if (isFinalOutputRecorded())
	    getFinalOutput().add(token);
	}

	// no token produced, ignore rest of actors
	if ((curr instanceof OutputProducer) && (token == null)) {
	  if (isDebugOn())
	    debug("No token generated, skipping rest of actors: " + curr.getFullName(), 2);
	  break;
	}
      }

      // all actors finished?
      if (isDebugOn())
	debug("notFinishedActor=" + notFinishedActor + ", pendingActors.size=" + pendingActors.size() + ", stopped=" + isStopped(), 2);
      finished = (notFinishedActor == null) && (pendingActors.size() == 0);
      if (isDebugOn())
	debug("---> execution finished: " + finished);
    }
    while (!finished || isStopped() || isStopping());

    return result;
  }

  /**
   * Executes the group of actors.
   *
   * @return		null if everything went smooth
   */
  public String execute() {
    String		result;
    String		msg;
    AbstractActor	start;

    result     = null;
    start      = null;
    m_Finished = false;
    m_Executed = true;

    if (m_ControlActor.getActorHandlerInfo().canContainStandalones()) {
      try {
	start = doExecuteStandalones();
	if (isDebugOn())
	  debug("doExecuteStandalones: " + ((start == null) ? "only standalones" : start.getFullName()));
      }
      catch (Exception e) {
	getSystemErr().printStackTrace(e);
	msg = e.toString();
	if (isDebugOn())
	  debug("doExecuteStandalones: " + msg);
	result = "Execution of standalones failed: " + msg;
      }
    }
    else {
      if (m_ControlActor.size() > 0)
	start = m_ControlActor.get(0);
    }

    // execute other actors until finished
    if ((result == null) && !isStopped() && !isStopping()) {
      if (start != null) {
	if (isDebugOn())
	  debug("doExecuteActors: start");
	try {
	  msg = doExecuteActors(start);
	  if (isDebugOn())
	    debug("doExecuteActors: " + ((msg == null) ? "OK" : msg));
	  if (msg != null)
	    result = "Execution of actors failed: " + msg;
	}
	catch (Exception e) {
	  getSystemErr().printStackTrace(e);
	  msg = e.toString();
	  if (isDebugOn())
	    debug("doExecuteActors: " + msg);
	  result = "Execution of actors died: " + msg;
	}
	if (isDebugOn())
	  debug("doExecuteActors: end");
      }
    }

    getFinalOutput().clear();

    m_Finished = true;

    return result;
  }

  /**
   * Checks whether the director has finished.
   *
   * @return		true if execution finished (or stopped)
   */
  public boolean isFinished() {
    return (m_Executed && m_Finished) || !m_Executed || m_Stopped;
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    super.stopExecution();

    getFinalOutput().clear();
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    super.cleanUp();

    if (m_FinalOutput != null)
      m_FinalOutput.clear();
  }
}
