/*
 * AbstractExternalActor.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.util.Vector;

import adams.core.Pausable;
import adams.core.Utils;
import adams.core.Variables;
import adams.core.io.FlowFile;
import adams.event.VariableChangeEvent;
import adams.event.VariableChangeEvent.Type;

/**
 * Ancestor of actors that load another actor from disk and execute it.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4165 $
 */
public abstract class AbstractExternalActor
  extends AbstractActor
  implements Pausable {

  /** for serialization. */
  private static final long serialVersionUID = 1024129351334661368L;

  /** the file the external actor is stored in. */
  protected FlowFile m_ActorFile;

  /** the external actor itself. */
  protected AbstractActor m_ExternalActor;

  /** indicates whether a variable is attached to the external file. */
  protected Boolean m_ActorFileIsVariable;

  /** the variable attached to the external file. */
  protected String m_ActorFileVariable;

  /** whether the external actor file has changed. */
  protected boolean m_ActorFileChanged;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "file", "actorFile",
	    new FlowFile("."));
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("actorFile");

    if (variable != null)
      return variable;
    else if (m_ActorFile != null)
      return m_ActorFile.toString();
    else
      return null;
  }

  /**
   * Sets the file containing the external actor.
   *
   * @param value	the actor file
   */
  public void setActorFile(FlowFile value) {
    m_ActorFile = value;
    reset();
  }

  /**
   * Returns the file containing the external actor.
   *
   * @return		the actor file
   */
  public FlowFile getActorFile() {
    return m_ActorFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorFileTipText() {
    return "The file containing the external actor.";
  }

  /**
   * Gets triggered when a variable changed (added, modified, removed).
   *
   * @param e		the event
   */
  public void variableChanged(VariableChangeEvent e) {
    super.variableChanged(e);

    if (m_ActorFileIsVariable == null) {
      m_ActorFileVariable   = getOptionManager().getVariableForProperty("actorFile");
      m_ActorFileIsVariable = (m_ActorFileVariable != null);
      if (m_ActorFileIsVariable != null)
	m_ActorFileVariable = Variables.extractName(m_ActorFileVariable);
    }

    if ((m_ActorFileIsVariable) && (e.getName().equals(m_ActorFileVariable)))
      m_ActorFileChanged = (e.getType() != Type.REMOVED);
  }

  /**
   * Returns the external actor.
   *
   * @return		the actor, can be null if not initialized yet or failed
   * 			to initialize
   */
  public AbstractActor getExternalActor() {
    return m_ExternalActor;
  }

  /**
   * Cleans up the external actor.
   */
  protected void cleanUpExternalActor() {
    if (m_ActorFileChanged && (m_ExternalActor != null)) {
      m_ExternalActor.wrapUp();
      m_ExternalActor.cleanUp();
      m_ExternalActor = null;
    }
  }

  /**
   * Sets up the external actor.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String setUpExternalActor() {
    String		result;
    Vector<String>	errors;

    result = null;

    if (!m_ActorFile.isFile()) {
      result = "'" + m_ActorFile.getAbsolutePath() + "' does not point to a file!";
    }
    else {
      errors = new Vector<String>();
      m_ExternalActor = ActorUtils.read(m_ActorFile.getAbsolutePath(), errors);
      if (!errors.isEmpty()) {
	result = "Error loading external actor '" + m_ActorFile.getAbsolutePath() + "':\n" + Utils.flatten(errors, "\n");
      }
      else if (m_ExternalActor == null) {
	result = "Error loading external actor '" + m_ActorFile.getAbsolutePath() + "'!";
      }
      else {
	m_ExternalActor.setParent(this);
	m_ExternalActor.setHeadless(isHeadless());
	result = m_ExternalActor.setUp();
      }
    }

    m_ActorFileChanged = false;

    return result;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;

    result = super.setUp();

    if (result == null) {
      // due to change in variable, we might need to clean up external actor
      cleanUpExternalActor();

      if (getOptionManager().getVariableForProperty("actorFile") == null) {
	if (m_ExternalActor == null)
	  result = setUpExternalActor();
      }
    }

    return result;
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    if ((m_ExternalActor != null) && (m_ExternalActor instanceof Pausable))
      ((Pausable) m_ExternalActor).pauseExecution();
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    if ((m_ExternalActor != null) && (m_ExternalActor instanceof Pausable))
      return ((Pausable) m_ExternalActor).isPaused();
    else
      return false;
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    if ((m_ExternalActor != null) && (m_ExternalActor instanceof Pausable))
      ((Pausable) m_ExternalActor).resumeExecution();
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    super.stopExecution();

    if (m_ExternalActor != null)
      m_ExternalActor.stopExecution();
  }

  /**
   * Gets called in the doExceute() method, after an optional
   * setUpExternalActor() call (in case a variable is used for the actor file),
   * but before the external actor's execute() method is called.
   * <p/>
   * Default implementation does nothing.
   *
   * @return		null if everything ok, otherwise error message
   * @see		#doExecute()
   * @see		#setUpExternalActor()
   */
  protected String preExecuteExternalActorHook() {
    return null;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;

    result = null;

    // not setup yet due to variable?
    cleanUpExternalActor();
    if (m_ExternalActor == null)
      result = setUpExternalActor();

    if (result == null)
      result = preExecuteExternalActorHook();

    if (result == null)
      result = m_ExternalActor.execute();

    return result;
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    if (m_ExternalActor != null)
      m_ExternalActor.wrapUp();

    m_ActorFileIsVariable = null;
    m_ActorFileVariable   = null;
    m_ActorFileChanged    = false;

    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void cleanUp() {
    if (m_ExternalActor != null) {
      m_ExternalActor.destroy();
      m_ExternalActor.setParent(null);
      m_ExternalActor = null;
    }

    super.cleanUp();
  }
}
