/*
 * AbstractPropertyUpdater.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.beans.PropertyDescriptor;

import adams.gui.goe.PropertyPath;

/**
 * Abstract ancestor for actors that manipulate properties of global actors,
 * e.g., WEKA classes.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3069 $
 */
public abstract class AbstractPropertyUpdater
  extends AbstractActor {

  /** for serialization. */
  private static final long serialVersionUID = 8068932300654252910L;

  /** the name of the global actor. */
  protected GlobalActorReference m_ActorName;

  /** the property path. */
  protected String m_Property;

  /** the global actor to update the property for. */
  protected transient AbstractActor m_GlobalActor;

  /** the property descriptor of the property to update. */
  protected transient PropertyDescriptor m_Descriptor;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "actor-name", "actorName",
	    new GlobalActorReference("unknown"));

    m_OptionManager.add(
	    "property", "property",
	    "");
  }

  /**
   * Sets the name of the actor to update.
   *
   * @param value	the name
   */
  public void setActorName(GlobalActorReference value) {
    m_ActorName = value;
    reset();
  }

  /**
   * Returns the name of the actor to update.
   *
   * @return		the name
   */
  public GlobalActorReference getActorName() {
    return m_ActorName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorNameTipText() {
    return "The name of the global actor to update the property for.";
  }

  /**
   * Sets the property to update.
   *
   * @param value	the property
   */
  public void setProperty(String value) {
    m_Property = value;
    reset();
  }

  /**
   * Returns the property to update.
   *
   * @return		the property
   */
  public String getProperty() {
    return m_Property;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String propertyTipText() {
    return "The property to update whenever the variable changes.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    return m_ActorName + "/" + m_Property;
  }

  /**
   * Updates the property.
   *
   * @param s		the string to set
   */
  protected void updateProperty(String s) {
    Object	value;

    value = PropertyHelper.convertVariableValue(m_Descriptor, s);

    // could we convert the value?
    if (value != null) {
      PropertyPath.setValue(m_GlobalActor, m_Property, value);
      if (isDebugOn())
	debug("Property '" + m_Property + "' of '" + m_ActorName + "' changed to: " + value);
    }
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;
    GlobalActorHelper	helper;
    Class		cls;

    result = super.setUp();

    if (result == null) {
      helper        = new GlobalActorHelper();
      m_GlobalActor = helper.findGlobalActorRecursive(this, m_ActorName);
      if (m_GlobalActor == null) {
	result = "Cannot find global actor '" + m_ActorName + "'!";
      }
      else {
	m_Descriptor = PropertyPath.getPropertyDescriptor(m_GlobalActor, m_Property);
	if (m_Descriptor == null) {
	  result = "Cannot find property '" + m_Property + "' in global actor '" + m_ActorName + "'!";
	}
	else {
	  cls = m_Descriptor.getReadMethod().getReturnType();
	  if (cls.isArray())
	    result = "Property '" + m_Property + "' is an array!";
	}
      }
    }

    return result;
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    super.wrapUp();

    m_GlobalActor = null;
  }
}
