/*
 * AbstractTemplate.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.util.Hashtable;

import adams.flow.template.AbstractActorTemplate;

/**
 * Abstract ancestor for all actors that use a template to generate the
 * actual actor/sub-flow to be executed.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4448 $
 */
public abstract class AbstractTemplate
  extends AbstractActor {

  /** for serialization. */
  private static final long serialVersionUID = 7246162048306571873L;

  /** the key for storing the current actor in the backup. */
  public final static String BACKUP_ACTOR = "actor";

  /** the template. */
  protected AbstractActorTemplate m_Template;

  /** the generated actor. */
  protected AbstractActor m_Actor;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "template", "template",
	    getDefaultTemplate());
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Actor = null;
  }

  /**
   * Returns the default template to use.
   * 
   * @return		the template
   */
  protected abstract AbstractActorTemplate getDefaultTemplate();
  
  /**
   * Sets the name of the global actor to use.
   *
   * @param value 	the global name
   */
  public void setTemplate(AbstractActorTemplate value) {
    m_Template = value;
    reset();
  }

  /**
   * Returns the name of the global actor in use.
   *
   * @return 		the global name
   */
  public AbstractActorTemplate getTemplate() {
    return m_Template;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String templateTipText() {
    return "The template to use for generating the actual actor.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("template");

    if (variable != null)
      return variable;
    else
      return m_Template.getClass().getSimpleName();
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_ACTOR);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_Actor != null)
      result.put(BACKUP_ACTOR, m_Actor);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_ACTOR)) {
      m_Actor = (AbstractActor) state.get(BACKUP_ACTOR);
      state.remove(BACKUP_ACTOR);
    }

    super.restoreState(state);
  }

  /**
   * Initializes the template for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String setUpTemplate() {
    String	result;

    result = null;

    m_Actor = m_Template.generate();
    if (m_Actor == null) {
      result = "Couldn't generate actor from template '" + getTemplate() + "'!";
    }
    else {
      m_Actor.setParent(this);
      m_Actor.setHeadless(isHeadless());
      result = m_Actor.setUp();
    }

    return result;
  }

  /**
   * Returns the actual actor that was generated from the template.
   * 
   * @return		the actual actor, null if not available
   */
  public AbstractActor getActualActor() {
    return m_Actor;
  }
  
  /**
   * Returns whether the actor has finished.
   *
   * @return		true if finished
   */
  public boolean isFinished() {
    if (m_Actor == null)
      return true;
    else
      return m_Actor.isFinished();
  }

  /**
   * Stops the execution. No message set.
   */
  public void stopExecution() {
    try {
      if (m_Actor != null)
	m_Actor.stopExecution();
    }
    catch (Exception e) {
      // ignored
    }

    super.stopExecution();
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    if (m_Actor != null)
      m_Actor.wrapUp();

    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void cleanUp() {
    super.cleanUp();

    if (m_Actor != null) {
      m_Actor.cleanUp();
      m_Actor = null;
    }
  }
}