/**
 * Screenshot.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.sink;

import java.io.File;
import java.util.Hashtable;

import javax.swing.JComponent;

import adams.core.VariableName;
import adams.core.Variables;
import adams.core.io.FileUtils;
import adams.core.io.PlaceholderFile;
import adams.flow.core.InputConsumer;
import adams.gui.print.NullWriter;

/**
 <!-- globalinfo-start -->
 * Actor that takes screenshots of graphical components.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Screenshot
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-title &lt;java.lang.String&gt; (property: title)
 * &nbsp;&nbsp;&nbsp;The title of the dialog.
 * &nbsp;&nbsp;&nbsp;default: Screenshot
 * </pre>
 *
 * <pre>-suffix &lt;java.lang.String&gt; (property: suffix)
 * &nbsp;&nbsp;&nbsp;An optional suffix for the filename, inserted before the extension.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 800
 * </pre>
 *
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 600
 * </pre>
 *
 * <pre>-output &lt;adams.core.io.PlaceholderFile&gt; (property: output)
 * &nbsp;&nbsp;&nbsp;The output directory.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-writer &lt;adams.gui.print.JComponentWriter [options]&gt; (property: writer)
 * &nbsp;&nbsp;&nbsp;The writer to use for generating the graphics output.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.print.NullWriter
 * </pre>
 *
 * <pre>-provider &lt;adams.flow.sink.DisplayPanelProvider [options]&gt; (property: panelProvider)
 * &nbsp;&nbsp;&nbsp;The actor for generating the display panels to take a screenshot of.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.sink.ClassifierErrors -writer adams.gui.print.NullWriter
 * </pre>
 *
 * <pre>-filename-prefix &lt;java.lang.String&gt; (property: filenamePrefix)
 * &nbsp;&nbsp;&nbsp;The prefix for the filename in case of auto-generation (no path, just name
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: screenshot
 * </pre>
 *
 * <pre>-filename-var &lt;java.lang.String&gt; (property: filenameVariable)
 * &nbsp;&nbsp;&nbsp;The variable to use for the filename instead of the auto-generated one.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3136 $
 */
public class Screenshot
  extends AbstractGraphicsGenerator {

  /** for serialization. */
  private static final long serialVersionUID = 4613925594824175758L;

  /** the key for storing the current counter in the backup. */
  public final static String BACKUP_COUNTER = "counter";

  /** the actor to use for generating panels. */
  protected DisplayPanelProvider m_PanelProvider;

  /** the prefix for the auto-generated filename. */
  protected String m_FilenamePrefix;

  /** the variable to use as filename. */
  protected VariableName m_FilenameVariable;

  /** the counter for the screenshots. */
  protected int m_Counter;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Actor that takes screenshots of graphical components.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "provider", "panelProvider",
	    new ImageViewer());

    m_OptionManager.add(
	    "filename-prefix", "filenamePrefix",
	    "screenshot");

    m_OptionManager.add(
	    "filename-var", "filenameVariable",
	    new VariableName());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;
    String	output;
    String	prefix;
    String	result;

    variable = getOptionManager().getVariableForProperty("filenameVariable");

    if (variable != null) {
      result = variable;
    }
    else if ((m_FilenameVariable != null) && (m_FilenameVariable.getValue().length() > 0)) {
      result = Variables.START + m_FilenameVariable + Variables.END;
    }
    else {
      variable = getOptionManager().getVariableForProperty("output");
      if (variable == null)
	output = m_Output.toString();
      else
	output = variable;

      variable = getOptionManager().getVariableForProperty("filenamePrefix");
      if (variable == null)
	prefix = m_FilenamePrefix.toString();
      else
	prefix = variable;

      if (!(getWriter() instanceof NullWriter)) {
	result = output + File.separator + prefix + "XYZ";
        if (getWriter().getExtensions().length > 0)
          result += getWriter().getExtensions()[0];
      }
      else {
	result = output;
      }
    }

    return result;
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_COUNTER);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_COUNTER, m_Counter);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_COUNTER)) {
      m_Counter = (Integer) state.get(BACKUP_COUNTER);
      state.remove(BACKUP_COUNTER);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Counter = 0;
  }

  /**
   * Returns the default title for the dialog.
   *
   * @return		the default title
   */
  protected String getDefaultTitle() {
    return "Screenshot";
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 800;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 600;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputTipText() {
    return "The output directory.";
  }

  /**
   * Sets the panel provider to use for generating the panels.
   *
   * @param value	the panel provider to use
   */
  public void setPanelProvider(DisplayPanelProvider value) {
    m_PanelProvider = value;
    reset();
  }

  /**
   * Returns the panel provider in use for generating the panels.
   *
   * @return		the panel provider in use
   */
  public DisplayPanelProvider getPanelProvider() {
    return m_PanelProvider;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String panelProviderTipText() {
    return "The actor for generating the display panels to take a screenshot of.";
  }

  /**
   * Sets the prefix for the filename in case of auto-generation.
   *
   * @param value	the prefix (just name, no path)
   */
  public void setFilenamePrefix(String value) {
    m_FilenamePrefix = value;
    reset();
  }

  /**
   * Returns the prefix for the filename in case of auto-generation.
   *
   * @return		the panel provider in use
   */
  public String getFilenamePrefix() {
    return m_FilenamePrefix;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String filenamePrefixTipText() {
    return "The prefix for the filename in case of auto-generation (no path, just name).";
  }

  /**
   * Sets the variable to use for generating the filename instead of the
   * auto-generated one.
   *
   * @param value	the variable name (without the @{ and })
   */
  public void setFilenameVariable(VariableName value) {
    m_FilenameVariable = value;
    reset();
  }

  /**
   * Returns the variable to use for generating the filename instead of the
   * auto-generated one.
   *
   * @return		the panel provider in use
   */
  public VariableName getFilenameVariable() {
    return m_FilenameVariable;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String filenameVariableTipText() {
    return "The variable to use for the filename instead of the auto-generated one.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->weka.classifiers.Evaluation.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    if ((m_PanelProvider != null) && (m_PanelProvider instanceof InputConsumer))
      return ((InputConsumer) m_PanelProvider).accepts();
    else
      return new Class[]{Object.class};
  }

  /**
   * Generates the component to display in the frame.
   *
   * @return		the component
   */
  protected JComponent generateComponent() {
    return m_PanelProvider.createDisplayPanel(m_InputToken);
  }

  /**
   * Generates the filename for the output.
   *
   * @return		the file
   */
  protected PlaceholderFile generateFilename() {
    PlaceholderFile	result;

    m_Counter++;

    if (getVariables().has(m_FilenameVariable.getValue())) {
      result = new PlaceholderFile(
	  getVariables().get(m_FilenameVariable.getValue()));
    }
    else {
      result = new PlaceholderFile(
	  m_Output.getAbsolutePath() + File.separator
	  + FileUtils.createFilename(m_FilenamePrefix + m_Counter + m_Writer.getExtensions()[0], "_"));
    }

    return result;
  }
}
