/*
 * Exec.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.source;

import adams.core.management.ProcessUtils;
import adams.core.option.OptionUtils;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Runs an external system command and broadcasts the generated output (stdout or stderr).
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - generates:<br/>
 * <pre>   java.lang.String</pre>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: Exec
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseString&gt; [-annotation ...] (property: annotations)
 *         The annotations to attach to this actor.
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-cmd &lt;java.lang.String&gt; (property: command)
 *         The external command to run.
 *         default: ls -l .
 * </pre>
 *
 * <pre>-stderr (property: outputStdErr)
 *         If set to true, then stderr is output instead of stdout.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4469 $
 */
public class Exec
  extends AbstractSource {

  /** for serialization. */
  private static final long serialVersionUID = -132045002653940359L;

  /** the command to run. */
  protected String m_Command;

  /** whether to output stderr instead of stdout. */
  protected boolean m_OutputStdErr;

  /** the output token to broadcast. */
  protected Token m_OutputToken;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Runs an external system command and broadcasts the generated output "
      + "(stdout or stderr).";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "cmd", "command",
	    "ls -l .");

    m_OptionManager.add(
	    "stderr", "outputStdErr",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("command");

    if (variable != null)
      return variable;
    else if ((m_Command != null) && (m_Command.length() != 0))
      return m_Command;
    else
      return null;
  }

  /**
   * Sets the command to run.
   *
   * @param value	the command
   */
  public void setCommand(String value) {
    m_Command = value;
    reset();
  }

  /**
   * Returns the command to run.
   *
   * @return 		the command
   */
  public String getCommand() {
    return m_Command;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String commandTipText() {
    return "The external command to run.";
  }

  /**
   * Sets whether to output stderr instead of stdout.
   *
   * @param value	if true then stderr is output instead of stdout
   */
  public void setOutputStdErr(boolean value) {
    m_OutputStdErr = value;
    reset();
  }

  /**
   * Returns whether stderr instead of stdout is output.
   *
   * @return 		true if stderr is output instead of stdout
   */
  public boolean getOutputStdErr() {
    return m_OutputStdErr;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String outputStdErrTipText() {
    return "If set to true, then stderr is output instead of stdout.";
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_OutputToken = null;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.String.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{String.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Object	output;

    result = null;

    try {
      output = ProcessUtils.execute(OptionUtils.splitOptions(m_Command), m_OutputStdErr);
      if (output instanceof Integer)
	result = "Command '" + m_Command + "' returned " + output;
      else
	m_OutputToken = new Token(output.toString());
    }
    catch (Exception e) {
      result = e.toString();
      getSystemErr().printStackTrace(e);
    }

    return result;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = m_OutputToken;
    m_OutputToken = null;

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_OutputToken != null);
  }
}
