/**
 * FlowTabbedPane.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.flow;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.tree.TreePath;

import adams.core.CleanUpHandler;
import adams.flow.core.AbstractActor;
import adams.gui.core.BaseTabbedPane;
import adams.gui.flow.tree.NodePopupMenuCustomizer;
import adams.gui.flow.tree.Tree;

/**
 * Specialized tabbed pane for Flow panels.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4419 $
 */
public class FlowTabbedPane
  extends BaseTabbedPane
  implements CleanUpHandler {

  /** for serialization. */
  private static final long serialVersionUID = -1675887825433207074L;

  /** the owning editor. */
  protected FlowEditorPanel m_Owner;

  /** the customizer for the node popup. */
  protected NodePopupMenuCustomizer m_NodePopupMenuCustomizer;

  /**
   * Initializes the tabbed pane.
   *
   * @param owner	the owning editor
   */
  public FlowTabbedPane(FlowEditorPanel owner) {
    super();

    m_Owner                   = owner;
    m_NodePopupMenuCustomizer = null;

    setTabLayoutPolicy(SCROLL_TAB_LAYOUT);
    setCloseTabsWithMiddelMouseButton(true);
    setMiddleMouseButtonCloseApprover(new MiddleMouseButtonCloseApprover() {
      public boolean approveClosingWithMiddleMouseButton(BaseTabbedPane source) {
	boolean	result = checkForModified();
	// to avoid second popup from checkModified() in removeTab method
	FlowPanel panel = getCurrentPanel();
	if (result && panel.isModified())
	  panel.setModified(false);
	return result;
      }
    });

    getModel().addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	// actor tabs
	if (getPanelCount() == 0)
	  m_Owner.getTabs().notifyTabs(
	      new TreePath[0],
	      new AbstractActor[0]);
	else
	  m_Owner.getTabs().notifyTabs(
	      m_Owner.getCurrentPanel().getTree().getSelectionPaths(),
	      m_Owner.getCurrentPanel().getTree().getSelectedActors());
	// title
	updateOwnerTitle();
	// update in general
	m_Owner.update();
      }
    });
  }

  /**
   * Returns the owning editor.
   *
   * @return		the owning editor
   */
  public FlowEditorPanel getOwner() {
    return m_Owner;
  }

  /**
   * Adds a new tab with an empty panel.
   *
   * @return		the new flow panel
   */
  public FlowPanel newPanel() {
    FlowPanel	result;

    result = new FlowPanel(this);
    addTab(result.getTitle(), result);
    setSelectedComponent(result);

    return result;
  }

  /**
   * Returns the number of panels in the tabbed pane.
   *
   * @return		the number of panels
   */
  public int getPanelCount() {
    return getTabCount();
  }

  /**
   * Returns the panel at the specified position.
   *
   * @param index	the tab index
   * @return		the requested panel
   */
  public FlowPanel getPanelAt(int index) {
    return (FlowPanel) getComponentAt(index);
  }

  /**
   * Returns if a panel is available.
   *
   * @return		true if a panel available
   */
  public boolean hasCurrentPanel() {
    return (getCurrentPanel() != null);
  }

  /**
   * Returns the currently selected panel.
   *
   * @return		the current panel, null if not available
   */
  public FlowPanel getCurrentPanel() {
    if (getSelectedIndex() != -1)
      return (FlowPanel) getComponentAt(getSelectedIndex());
    else
      return null;
  }

  /**
   * Returns the tree at the specified position.
   *
   * @param index	the tab index
   * @return		the requested tree
   */
  public Tree getTreeAt(int index) {
    return getPanelAt(index).getTree();
  }

  /**
   * Returns whether a tree is currently selected.
   *
   * @return		true if a tree available
   */
  public boolean hasCurrentTree() {
    return (getCurrentTree() != null);
  }

  /**
   * Returns the currently selected tree.
   *
   * @return		the currently selected tree, null if not available
   */
  public Tree getCurrentTree() {
    if (getSelectedIndex() != -1)
      return getPanelAt(getSelectedIndex()).getTree();
    else
      return null;
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    int		i;

    for (i = 0; i < getPanelCount(); i++)
      getPanelAt(i).cleanUp();
  }

  /**
   * Sets the node popup menu customizer to use.
   *
   * @param value	the customizer, null to disable
   */
  public void setNodePopupMenuCustomizer(NodePopupMenuCustomizer value) {
    int		i;

    m_NodePopupMenuCustomizer = value;

    for (i = 0; i < getPanelCount(); i++)
      getPanelAt(i).getTree().setNodePopupMenuCustomizer(value);
  }

  /**
   * Returns the current node popup menu customizer in use, if any.
   *
   * @return		the customizer, can be null if default used
   */
  public NodePopupMenuCustomizer getNodePopupMenuCustomizer() {
    return m_NodePopupMenuCustomizer;
  }

  /**
   * Returns whether we can proceed with the operation or not, depending on
   * whether the user saved the flow or discarded the changes.
   *
   * @return		true if safe to proceed
   */
  protected boolean checkForModified() {
    if (m_Owner == null)
      return true;
    return m_Owner.checkForModified();
  }

  /**
   * Hook method that gets executed after a tab was successfully removed with
   * a middle mouse button click.
   */
  protected void afterTabClosedWithMiddleMouseButton() {
    updateOwnerTitle();
  }

  /**
   * Updates the owner's title.
   */
  protected void updateOwnerTitle() {
    String	title;

    if (m_Owner == null)
      return;

    if (getPanelCount() == 0) {
      title = FlowEditorPanel.DEFAULT_TITLE;
      m_Owner.setParentTitle(title);
    }
    else {
      getCurrentPanel().updateTitle();
    }
  }

  /**
   * Removes the tab at <code>index</code>.
   * After the component associated with <code>index</code> is removed,
   * its visibility is reset to true to ensure it will be visible
   * if added to other containers.
   *
   * @param index the index of the tab to be removed
   */
  public void removeTabAt(int index) {
    FlowPanel	panel;

    if (index < 0)
      return;
    if (!checkForModified())
      return;

    panel = getPanelAt(index);
    panel.cleanUp();

    super.removeTabAt(index);

    updateOwnerTitle();
  }
}
